/**
 * @file
 * Loads and renders external JavaScript components.
 *
 * Handles recursive rendering of nested components by re-attaching behaviors
 * after each component renders.
 */

((Drupal, once) => {
  const bundleCache = new Map();

  /**
   * Loads JavaScript bundles as ES modules.
   *
   * @param {string[]} urls
   *   Array of bundle URLs to load.
   *
   * @return {Promise<Module[]>}
   *   Promise resolving to array of loaded modules.
   */
  async function loadBundles(urls) {
    return Promise.all(
      urls.map((url) => {
        if (!bundleCache.has(url)) {
          bundleCache.set(url, import(url));
        }
        return bundleCache.get(url);
      }),
    );
  }

  /**
   * Renders a single component.
   *
   * @param {HTMLElement} container
   *   The component container element.
   *
   * @return {Object}
   *   Result object with {success, element} or {error}.
   */
  async function renderComponent(container) {
    const componentName = container.dataset.componentName;
    const props = JSON.parse(container.dataset.componentProps || '{}');
    const slots = JSON.parse(container.dataset.componentSlots || '{}');
    const bundles = JSON.parse(container.dataset.componentBundles);

    const modules = await loadBundles(bundles);
    const mainModule = modules[modules.length - 1];

    const renderFn = mainModule.render || mainModule.default;
    if (!renderFn) {
      return { error: Drupal.t('Bundle does not export render function') };
    }

    const result = await renderFn(container, componentName, props, slots);

    if (result === null) {
      return {
        error: Drupal.t("Component '@name' not found in bundle", {
          '@name': componentName,
        }),
      };
    }

    return { success: true, element: result };
  }

  Drupal.behaviors.canvasExtjsComponentLoader = {
    attach(context) {
      once('extjs-component', '.extjs-component-container', context).forEach(
        async (container) => {
          try {
            const bundles = JSON.parse(
              container.dataset.componentBundles || '[]',
            );
            await loadBundles(bundles);

            const result = await renderComponent(container);

            if (result.error) {
              container.classList.add('error');
              container.textContent = Drupal.t('Error: @message', {
                '@message': result.error,
              });
              return;
            }

            // Wait for DOM updates before re-attaching behaviors for nested components.
            await new Promise((resolve) => {
              requestAnimationFrame(() =>
                requestAnimationFrame(() => resolve()),
              );
            });

            // Re-attach behaviors to process nested containers in slots.
            Drupal.attachBehaviors(container);
          } catch (error) {
            container.classList.add('error');
            container.textContent = Drupal.t('Render failed: @message', {
              '@message': error.message,
            });
            console.error('Component render error:', error);
          }
        },
      );
    },
  };
})(Drupal, once);
