<?php

declare(strict_types=1);

namespace Drupal\Tests\canvas_extjs\Kernel\Component;

use Drupal\canvas\Entity\Component as ComponentEntity;
use Drupal\canvas\Entity\VersionedConfigEntityBase;
use Drupal\canvas\PropExpressions\StructuredData\EvaluationResult;
use Drupal\canvas_extjs\Plugin\Canvas\ComponentSource\ExternalJavaScriptComponent;
use Drupal\custom_elements\CustomElement;
use Drupal\KernelTests\KernelTestBase;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the ExternalJavaScriptComponent source plugin.
 */
#[Group("canvas_extjs")]
class ExternalJavaScriptComponentTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'canvas',
    'canvas_extjs',
    'custom_elements',
    'field',
    'text',
    'user',
    'options',
    'filter',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['system']);
    $this->installEntitySchema('user');
    $this->installEntitySchema('component');
    // Don't install canvas config to avoid editor dependencies.
  }

  /**
   * Tests creating a component entity using the helper method.
   */
  public function testCreateTestComponentEntities(): void {
    ExternalJavaScriptComponent::createTestComponentEntities('TestButton');

    // Load the created component (now using snake_case ID).
    $component = ComponentEntity::load('extjs.test_button');
    $this->assertNotNull($component);
    $this->assertEquals('extjs.test_button', $component->id());
    $this->assertEquals('Test Button', $component->label());
    $this->assertEquals('Vue Components', $component->get('category'));
    $this->assertEquals('extjs', $component->get('source'));
    $this->assertEquals('TestButton', $component->get('source_local_id'));
    $this->assertEquals('v1', $component->getActiveVersion());
    $this->assertTrue($component->status());

    // Verify prop_field_definitions were generated.
    $component->loadVersion('v1');
    $settings = $component->getSettings();
    $this->assertArrayHasKey('label', $settings['prop_field_definitions']);
    $this->assertArrayHasKey('variant', $settings['prop_field_definitions']);
    $this->assertEquals('string', $settings['prop_field_definitions']['label']['field_type']);
    $this->assertEquals('string_textfield', $settings['prop_field_definitions']['label']['field_widget']);
  }

  /**
   * Tests creating a custom component entity manually.
   */
  public function testCreateCustomComponentEntity(): void {
    // Create a custom component with different props.
    $settings = [
      'label' => 'Custom Button',
      'local_source_id' => 'custom-button',
      'props' => [
        'type' => 'object',
        'required' => ['text'],
        'properties' => [
          'text' => [
            'type' => 'string',
            'title' => 'Button Text',
            'examples' => ['Click me!'],
          ],
          'variant' => [
            'type' => 'string',
            'title' => 'Button Variant',
            'enum' => ['primary', 'secondary', 'danger'],
            'examples' => ['primary'],
          ],
          'disabled' => [
            'type' => 'boolean',
            'title' => 'Disabled',
            'examples' => [FALSE],
          ],
        ],
      ],
      'slots' => [
        'icon' => [
          'title' => 'Icon Slot',
          'description' => 'Optional icon to display in the button',
        ],
      ],
      'prop_field_definitions' => [],
    ];

    $component = ComponentEntity::create([
      'id' => 'extjs.custom_button',
      'label' => 'Custom Button',
      'category' => 'Custom Components',
      'source' => ExternalJavaScriptComponent::SOURCE_PLUGIN_ID,
      'provider' => NULL,
      'source_local_id' => 'custom-button',
      'active_version' => 'v1',
      'versioned_properties' => [
        VersionedConfigEntityBase::ACTIVE_VERSION => ['settings' => $settings],
      ],
      'status' => TRUE,
    ]);

    // Generate prop_field_definitions before saving.
    ExternalJavaScriptComponent::ensurePropFieldDefinitions($component);
    $component->save();

    // Reload to verify persistence.
    $loaded_component = ComponentEntity::load('extjs.custom_button');
    $this->assertNotNull($loaded_component);
    $this->assertEquals('Custom Button', $loaded_component->label());
    $this->assertEquals('Custom Components', $loaded_component->get('category'));

    // Verify all props have field definitions.
    $loaded_component->loadVersion('v1');
    $loaded_settings = $loaded_component->getSettings();
    $this->assertArrayHasKey('prop_field_definitions', $loaded_settings);
    $this->assertArrayHasKey('text', $loaded_settings['prop_field_definitions']);
    $this->assertArrayHasKey('variant', $loaded_settings['prop_field_definitions']);
    $this->assertArrayHasKey('disabled', $loaded_settings['prop_field_definitions']);

    // Verify field types.
    $this->assertEquals('string', $loaded_settings['prop_field_definitions']['text']['field_type']);
    $this->assertEquals('string_textfield', $loaded_settings['prop_field_definitions']['text']['field_widget']);
    $this->assertEquals('list_string', $loaded_settings['prop_field_definitions']['variant']['field_type']);
    $this->assertEquals('boolean', $loaded_settings['prop_field_definitions']['disabled']['field_type']);

    // Verify slots are preserved.
    $this->assertArrayHasKey('icon', $loaded_settings['slots']);
    $this->assertEquals('Icon Slot', $loaded_settings['slots']['icon']['title']);
  }

  /**
   * Tests that TwoColumnLayout component has slots defined.
   */
  public function testTwoColumnLayoutSlotsRegistration(): void {
    // Create TwoColumnLayout component.
    ExternalJavaScriptComponent::createTestComponentEntities('TwoColumnLayout');

    // Load the component.
    $component = ComponentEntity::load('extjs.two_column_layout');
    $this->assertNotNull($component);
    $this->assertEquals('Two Column Layout', $component->label());

    // Load settings and verify slots.
    $component->loadVersion('v1');
    $settings = $component->getSettings();
    $this->assertArrayHasKey('slots', $settings);
    $this->assertArrayHasKey('column-one', $settings['slots']);
    $this->assertArrayHasKey('column-two', $settings['slots']);

    // Verify slot definitions.
    $this->assertEquals('Column One', $settings['slots']['column-one']['title']);
    $this->assertEquals('The contents of the first column.', $settings['slots']['column-one']['description']);
    $this->assertEquals('Column Two', $settings['slots']['column-two']['title']);
    $this->assertEquals('The contents of the second column.', $settings['slots']['column-two']['description']);
  }

  /**
   * Tests that slots are passed through data-component-data when rendering.
   */
  public function testSlotsPassedThroughDataComponentData(): void {
    ExternalJavaScriptComponent::createTestComponentEntities('TwoColumnLayout');
    $component = ComponentEntity::load('extjs.two_column_layout');
    $this->assertNotNull($component);
    $component->loadVersion('v1');
    $layout_source = $component->getComponentSource();
    $this->assertInstanceOf(ExternalJavaScriptComponent::class, $layout_source);

    // Build inputs for renderComponent.
    $inputs = [
      'props' => [
        'width' => new EvaluationResult(33),
      ],
    ];
    $build = $layout_source->renderComponent($inputs, [], 'test-uuid-123', TRUE);
    $this->assertIsArray($build);
    // Debug: Check if build has content.
    $this->assertNotEmpty($build, 'Build array should not be empty');

    // Add slots with simple HTML markup (using kebab-case names).
    $slots = [
      'column-one' => ['#markup' => '<p>Left column content</p>'],
      'column-two' => ['#markup' => '<p>Right column content</p>'],
    ];
    $layout_source->setSlots($build, $slots);

    // Verify slots are stored in #slots before rendering.
    $this->assertArrayHasKey('#slots', $build);
    $this->assertArrayHasKey('column-one', $build['#slots']);
    $this->assertArrayHasKey('column-two', $build['#slots']);

    // Render to trigger #pre_render callback which processes slots.
    $renderer = \Drupal::service('renderer');
    $html = (string) $renderer->renderInIsolation($build);
    // Then verify the custom-element is prepared correctly for rendering.
    $custom_element = $build['#custom_element'];
    $this->assertInstanceOf(CustomElement::class, $custom_element);
    $element_slots = $custom_element->getSlots();
    $this->assertArrayHasKey('column-one', $element_slots);
    $this->assertArrayHasKey('column-two', $element_slots);
    // Verify html output is good also.
    $this->assertNotEmpty($html, 'HTML should not be empty');
    $this->assertStringContainsString('TwoColumnLayout', $html);
    $this->assertStringContainsString('Left column content', $html);
    $this->assertStringContainsString('Right column content', $html);

    // Test nested component in slot.
    ExternalJavaScriptComponent::createTestComponentEntities('TestButton');
    $button_component = ComponentEntity::load('extjs.test_button');
    $button_component->loadVersion('v1');
    $button_source = $button_component->getComponentSource();
    $button_build = $button_source->renderComponent([
      'props' => [
        'label' => new EvaluationResult('Nested Button'),
        'variant' => new EvaluationResult('success'),
      ],
    ], [], 'button-uuid', TRUE);

    // Create a new layout with the button render array as a slot.
    $layout_build2 = $layout_source->renderComponent([
      'props' => ['width' => new EvaluationResult(50)],
    ], [], 'layout-uuid-2', TRUE);
    $layout_source->setSlots($layout_build2, [
      'column-one' => $button_build,
      'column-two' => ['#markup' => '<p>Right column</p>'],
    ]);

    // Render to trigger #pre_render callback which processes slots.
    // Then verify the custom-element is prepared correctly for rendering.
    $renderer->renderInIsolation($layout_build2);

    // Check the nested structure of the CustomElement by inspecting it's
    // toArray() conversion as used when normalized to JSON.
    $layout_element = $layout_build2['#custom_element'];
    $layout_array = $layout_element->toArray(FALSE, NULL, TRUE);
    $this->assertArrayHasKey('element', $layout_array);
    $this->assertEquals('TwoColumnLayout', $layout_array['element']);
    // Verify props are in the props object.
    $this->assertArrayHasKey('props', $layout_array);
    $this->assertArrayHasKey('width', $layout_array['props']);
    $this->assertEquals(50, $layout_array['props']['width']);
    // Note: toArray() uses explicit format with slots separated from props.
    // Slot names are converted from kebab-case to camelCase.
    $this->assertArrayHasKey('slots', $layout_array, 'Layout array should have slots key');
    $this->assertArrayHasKey('columnOne', $layout_array['slots'], 'Layout array should have columnOne slot');
    $this->assertArrayHasKey('columnTwo', $layout_array['slots'], 'Layout array should have columnTwo slot');

    // Verify the nested component structure in columnOne slot.
    // The slot contains the nested element directly (not wrapped in an array).
    $column_one_slot = $layout_array['slots']['columnOne'];
    $this->assertIsArray($column_one_slot, 'Slot should be an array containing nested element');
    $this->assertArrayHasKey('element', $column_one_slot, 'Slot should have element key');
    $this->assertEquals('TestButton', $column_one_slot['element'], 'Nested element should be TestButton');
    // Props are also in a props object in the nested element.
    $this->assertArrayHasKey('props', $column_one_slot, 'Nested element should have props');
    $this->assertArrayHasKey('variant', $column_one_slot['props'], 'Nested button should have variant prop');
    $this->assertEquals('success', $column_one_slot['props']['variant']);
  }

}
