<?php

declare(strict_types=1);

namespace Drupal\Tests\canvas_extjs\Kernel;

use Drupal\canvas_extjs\ComponentIndexManager;
use Drupal\KernelTests\KernelTestBase;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the ComponentIndexManager service.
 */
#[Group("canvas_extjs")]
class ComponentIndexManagerTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'canvas',
    'canvas_extjs',
    'custom_elements',
    'field',
    'text',
    'user',
    'options',
    'filter',
  ];

  /**
   * The component index manager service.
   *
   * @var \Drupal\canvas_extjs\ComponentIndexManager
   */
  protected ComponentIndexManager $componentIndexManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig(['system']);
    $this->installEntitySchema('user');
    $this->installEntitySchema('component');

    $this->componentIndexManager = $this->container->get('canvas_extjs.component_index_manager');
  }

  /**
   * Tests validation with valid component-index.json fixture.
   */
  public function testValidationWithValidFixture(): void {
    // Load the fixture.
    $fixturePath = __DIR__ . '/../../fixtures/component-index.json';
    $this->assertFileExists($fixturePath, 'Fixture file exists');

    $fixtureContent = file_get_contents($fixturePath);
    $data = json_decode($fixtureContent, TRUE);
    $this->assertIsArray($data, 'Fixture JSON is valid');

    // Validate the fixture data - should not throw.
    $this->componentIndexManager->validate($data);

    $this->assertArrayHasKey('version', $data);
    $this->assertEquals('1.0', $data['version']);
    $this->assertArrayHasKey('components', $data);
    $this->assertCount(5, $data['components'], 'Fixture contains 5 components');
  }

  /**
   * Tests validation with invalid data (missing required fields).
   */
  public function testValidationWithInvalidData(): void {
    // Missing required 'version' field.
    $invalidData = [
      'components' => [],
    ];

    $this->expectException(\RuntimeException::class);
    $this->expectExceptionMessage('Validation failed');
    $this->componentIndexManager->validate($invalidData);
  }

  /**
   * Tests validation with missing components field.
   */
  public function testValidationMissingComponents(): void {
    $invalidData = [
      'version' => '1.0',
    ];

    $this->expectException(\RuntimeException::class);
    $this->expectExceptionMessage('Validation failed');
    $this->componentIndexManager->validate($invalidData);
  }

  /**
   * Tests validation with component missing required id field.
   */
  public function testValidationComponentMissingId(): void {
    $invalidData = [
      'version' => '1.0',
      'components' => [
        [
          'name' => 'Test Component',
          'category' => 'Test',
        ],
      ],
    ];

    $this->expectException(\RuntimeException::class);
    $this->expectExceptionMessage('Validation failed');
    $this->componentIndexManager->validate($invalidData);
  }

  /**
   * Tests fetching from local file path.
   */
  public function testFetchFromLocalFile(): void {
    $fixturePath = __DIR__ . '/../../fixtures/component-index.json';

    $data = $this->componentIndexManager->fetch($fixturePath);

    $this->assertIsArray($data);
    $this->assertArrayHasKey('version', $data);
    $this->assertArrayHasKey('components', $data);
    $this->assertCount(5, $data['components']);
  }

  /**
   * Tests fetching from non-existent file.
   */
  public function testFetchFromNonExistentFile(): void {
    $this->expectException(\RuntimeException::class);
    $this->expectExceptionMessage('not found');

    $this->componentIndexManager->fetch('/non/existent/path.json');
  }

  /**
   * Tests fetching invalid JSON.
   */
  public function testFetchInvalidJson(): void {
    // Create a temporary file with invalid JSON.
    $tempFile = sys_get_temp_dir() . '/invalid-component-index.json';
    file_put_contents($tempFile, '{invalid json}');

    $this->expectException(\RuntimeException::class);
    $this->expectExceptionMessage('Invalid JSON');

    try {
      $this->componentIndexManager->fetch($tempFile);
    }
    finally {
      unlink($tempFile);
    }
  }

  /**
   * Tests fetching JSON that fails validation.
   */
  public function testFetchInvalidComponentIndex(): void {
    // Create a temporary file with valid JSON but invalid component index.
    $tempFile = sys_get_temp_dir() . '/invalid-structure.json';
    file_put_contents($tempFile, json_encode(['version' => '1.0']));

    $this->expectException(\RuntimeException::class);
    $this->expectExceptionMessage('Validation failed');

    try {
      $this->componentIndexManager->fetch($tempFile);
    }
    finally {
      unlink($tempFile);
    }
  }

  /**
   * Tests registering components from a component index.
   */
  public function testRegisterComponents(): void {
    $fixturePath = __DIR__ . '/../../fixtures/component-index.json';

    $result = $this->componentIndexManager->register($fixturePath);

    // Verify result structure.
    $this->assertArrayHasKey('created', $result);
    $this->assertArrayHasKey('updated', $result);
    $this->assertArrayHasKey('errors', $result);

    // All 5 components should be created.
    $this->assertCount(5, $result['created'], 'Expected 5 components created. Errors: ' . print_r($result['errors'], TRUE));
    $this->assertEmpty($result['updated']);
    $this->assertEmpty($result['errors']);

    // Verify one of the created components.
    $storage = $this->container->get('entity_type.manager')->getStorage('component');
    /** @var \Drupal\canvas\Entity\ComponentInterface $component */
    $component = $storage->load('extjs.test_button');

    $this->assertNotNull($component);
    $this->assertEquals('Test Button', $component->label());
    $this->assertEquals('JS Components', $component->get('category'));

    // Verify source_url is stored in settings.
    $component->loadVersion($component->getActiveVersion());
    $settings = $component->getSettings();
    $this->assertEquals($fixturePath, $settings['source_url']);
    $this->assertArrayHasKey('props', $settings);
    $this->assertArrayHasKey('prop_field_definitions', $settings);
  }

  /**
   * Tests re-registering (updating) components.
   */
  public function testReRegisterComponents(): void {
    $fixturePath = __DIR__ . '/../../fixtures/component-index.json';

    // First registration.
    $result1 = $this->componentIndexManager->register($fixturePath);
    $this->assertCount(5, $result1['created']);

    // Second registration should update.
    $result2 = $this->componentIndexManager->register($fixturePath);
    $this->assertEmpty($result2['created']);
    $this->assertCount(5, $result2['updated']);
    $this->assertEmpty($result2['errors']);
  }

  /**
   * Tests unregistering components from a source.
   */
  public function testUnregisterComponents(): void {
    $fixturePath = __DIR__ . '/../../fixtures/component-index.json';

    // Register components first.
    $this->componentIndexManager->register($fixturePath);

    // Unregister them.
    $result = $this->componentIndexManager->unregister($fixturePath);

    $this->assertArrayHasKey('deleted', $result);
    $this->assertArrayHasKey('errors', $result);
    $this->assertCount(5, $result['deleted']);
    $this->assertEmpty($result['errors']);

    // Verify components are deleted.
    $storage = $this->container->get('entity_type.manager')->getStorage('component');
    $component = $storage->load('extjs.test_button');
    $this->assertNull($component);
  }

  /**
   * Tests unregistering doesn't affect components from other sources.
   */
  public function testUnregisterOnlyAffectsSpecificSource(): void {
    $fixturePath = __DIR__ . '/../../fixtures/component-index.json';

    // Register from fixture.
    $this->componentIndexManager->register($fixturePath);

    // Create a second component index with different source.
    $otherPath = sys_get_temp_dir() . '/other-component-index.json';
    $otherIndexJson = <<<'JSON'
{
  "version": "1.0",
  "components": [
    {
      "id": "OtherComponent",
      "name": "Other Component",
      "category": "Test",
      "props": {
        "type": "object",
        "properties": {}
      }
    }
  ]
}
JSON;
    file_put_contents($otherPath, $otherIndexJson);

    try {
      // Register from other source.
      $this->componentIndexManager->register($otherPath);

      // Unregister fixture components.
      $result = $this->componentIndexManager->unregister($fixturePath);

      $this->assertCount(5, $result['deleted']);

      // Verify other component still exists.
      $storage = $this->container->get('entity_type.manager')->getStorage('component');
      $storage->resetCache();
      $otherComponent = $storage->load('extjs.other_component');
      $this->assertNotNull($otherComponent, 'Component from other source should not be deleted');
    }
    finally {
      unlink($otherPath);
    }
  }

}
