<?php

declare(strict_types=1);

namespace Drupal\canvas_extjs\Commands;

use Drupal\canvas_extjs\ComponentIndexManager;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;

/**
 * Drush commands for Canvas External JS components.
 */
final class CanvasExtjsCommands extends DrushCommands {

  /**
   * Constructs a CanvasExtjsCommands object.
   *
   * @param \Drupal\canvas_extjs\ComponentIndexManager $componentIndexManager
   *   The component index manager service.
   */
  public function __construct(
    protected ComponentIndexManager $componentIndexManager,
  ) {
    parent::__construct();
  }

  /**
   * Builds register options from drush command options.
   *
   * @param array $options
   *   Drush command options (same structure as register() $options).
   *
   * @return array
   *   Options array for ComponentIndexManager::register().
   */
  protected function buildRegisterOptions(array $options): array {
    $registerOptions = [];

    // Check for conflicting options.
    if ($options['custom-elements-preview-provider'] && $options['javascript']) {
      throw new \InvalidArgumentException(
        'Cannot use both --custom-elements-preview-provider and --javascript options'
      );
    }

    if ($options['custom-elements-preview-provider']) {
      if ($options['custom-elements-preview-provider'] === 'auto') {
        $registerOptions['custom_elements_preview'] = ['preview_provider' => 'auto'];
      }
      else {
        $registerOptions['custom_elements_preview'] = [
          'preview_provider' => $options['custom-elements-preview-provider'],
        ];

        if ($options['base-url']) {
          $registerOptions['custom_elements_preview']['base_url'] = $options['base-url'];
        }
      }
    }
    elseif ($options['javascript']) {
      $urls = array_map('trim', explode(',', $options['javascript']));
      if (empty($urls)) {
        throw new \InvalidArgumentException(
          '--javascript requires at least one URL'
        );
      }
      $registerOptions['javascript'] = $urls;
    }

    return $registerOptions;
  }

  /**
   * Register external JavaScript components from a component index.
   */
  #[CLI\Command(name: 'canvas:extjs-register', aliases: ['cxr'])]
  #[CLI\Argument(name: 'source', description: 'URL or file path to component-index.json')]
  #[CLI\Option(name: 'custom-elements-preview-provider', description: 'Preview provider plugin ID ("auto" for automatic selection, or specific like "nuxt")')]
  #[CLI\Option(name: 'base-url', description: 'If a specific custom elements preview provider is used, the base URL for it.')]
  #[CLI\Option(name: 'javascript', description: 'Comma-separated JavaScript bundle URLs to load for rendering.')]
  #[CLI\Usage(name: 'canvas:extjs-register https://example.com/component-index.json', description: 'Register with default preview (auto)')]
  #[CLI\Usage(name: 'canvas:extjs-register https://example.com/index.json --custom-elements-preview-provider=nuxt --base-url=http://localhost:3000', description: 'Register with Nuxt preview provider')]
  #[CLI\Usage(name: 'canvas:extjs-register https://example.com/index.json --javascript=https://cdn.example.com/bundle.js', description: 'Register with JavaScript bundle for rendering')]
  public function register(
    string $source,
    array $options = [
      'base-url' => NULL,
      'custom-elements-preview-provider' => NULL,
      'javascript' => NULL,
    ],
  ): int {
    $this->io()->title('Registering External JS Components');
    $this->io()->text(sprintf('Source: %s', $source));

    $registerOptions = $this->buildRegisterOptions($options);

    try {
      $result = $this->componentIndexManager->register($source, $registerOptions);

      // Display results.
      if (!empty($result['created'])) {
        $this->io()->success(sprintf('Created %d component(s):', count($result['created'])));
        foreach ($result['created'] as $id) {
          $this->io()->text('  - ' . $id);
        }
      }

      if (!empty($result['updated'])) {
        $this->io()->success(sprintf('Updated %d component(s):', count($result['updated'])));
        foreach ($result['updated'] as $id) {
          $this->io()->text('  - ' . $id);
        }
      }

      if (!empty($result['errors'])) {
        $this->io()->warning(sprintf('Failed to register %d component(s):', count($result['errors'])));
        foreach ($result['errors'] as $error) {
          $this->io()->text('  - ' . $error);
        }
        return DrushCommands::EXIT_FAILURE_WITH_CLARITY;
      }

      $total = count($result['created']) + count($result['updated']);
      if ($total === 0) {
        $this->io()->note('No components were registered.');
        return DrushCommands::EXIT_SUCCESS;
      }

      $this->io()->success(sprintf('Successfully registered %d component(s).', $total));
      return DrushCommands::EXIT_SUCCESS;
    }
    catch (\Exception $e) {
      $this->io()->error('Failed to register components: ' . $e->getMessage());
      return DrushCommands::EXIT_FAILURE;
    }
  }

  /**
   * Unregister external JavaScript components from a source.
   */
  #[CLI\Command(name: 'canvas:extjs-unregister', aliases: ['cxu'])]
  #[CLI\Argument(name: 'source', description: 'URL or file path that was used during registration')]
  #[CLI\Usage(name: 'canvas:extjs-unregister https://example.com/component-index.json', description: 'Unregister components from this source')]
  #[CLI\Usage(name: 'canvas:extjs-unregister /path/to/component-index.json', description: 'Unregister components from local file source')]
  public function unregister(string $source): int {
    $this->io()->title('Unregistering External JS Components');
    $this->io()->text(sprintf('Source: %s', $source));

    try {
      $result = $this->componentIndexManager->unregister($source);

      if (!empty($result['deleted'])) {
        $this->io()->success(sprintf('Deleted %d component(s):', count($result['deleted'])));
        foreach ($result['deleted'] as $id) {
          $this->io()->text('  - ' . $id);
        }
      }

      if (!empty($result['errors'])) {
        $this->io()->warning(sprintf('Failed to unregister %d component(s):', count($result['errors'])));
        foreach ($result['errors'] as $error) {
          $this->io()->text('  - ' . $error);
        }
        return DrushCommands::EXIT_FAILURE_WITH_CLARITY;
      }

      if (empty($result['deleted'])) {
        $this->io()->note('No components found from this source.');
        return DrushCommands::EXIT_SUCCESS;
      }

      $this->io()->success(sprintf('Successfully unregistered %d component(s).', count($result['deleted'])));
      return DrushCommands::EXIT_SUCCESS;
    }
    catch (\Exception $e) {
      $this->io()->error('Failed to unregister components: ' . $e->getMessage());
      return DrushCommands::EXIT_FAILURE;
    }
  }

}
