<?php

declare(strict_types=1);

namespace Drupal\canvas_extjs\Plugin\Canvas\ComponentSource;

use Drupal\canvas\Attribute\ComponentSource;
use Drupal\canvas\ComponentDoesNotMeetRequirementsException;
use Drupal\canvas\Entity\Component as ComponentEntity;
use Drupal\canvas\Entity\ComponentInterface;
use Drupal\canvas\Entity\VersionedConfigEntityBase;
use Drupal\canvas\Plugin\Canvas\ComponentSource\GeneratedFieldExplicitInputUxComponentSourceBase;
use Drupal\canvas\PropShape\PropShapeRepositoryInterface;
use Drupal\canvas\ShapeMatcher\PropSourceSuggester;
use Drupal\canvas_extjs\CamelCaseSnakeCaseConverterTrait;
use Drupal\Component\Utility\Html;
use Drupal\Core\Render\Markup;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\WidgetPluginManager;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Plugin\Component as ComponentPlugin;
use Drupal\Core\Plugin\Component as SdcPlugin;
use Drupal\Core\Security\TrustedCallbackInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Theme\Component\ComponentValidator;
use Drupal\custom_elements\CustomElement;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a generic JavaScript component source.
 *
 * @phpstan-ignore-next-line classExtendsInternalClass.classExtendsInternalClass
 */
#[ComponentSource(
  id: self::SOURCE_PLUGIN_ID,
  label: new TranslatableMarkup('External JavaScript Components'),
  supportsImplicitInputs: FALSE,
  discovery: FALSE,
)]
final class ExternalJavaScriptComponent extends GeneratedFieldExplicitInputUxComponentSourceBase implements TrustedCallbackInterface {

  use CamelCaseSnakeCaseConverterTrait;

  public const SOURCE_PLUGIN_ID = 'extjs';

  /**
   * Constructs a new JavaScriptComponent.
   *
   * @param array $configuration
   *   Configuration.
   * @param string $plugin_id
   *   Plugin ID.
   * @param array $plugin_definition
   *   Plugin definition.
   * @param \Drupal\Core\Theme\Component\ComponentValidator $componentValidator
   *   Component validator.
   * @param \Drupal\Core\Field\WidgetPluginManager $fieldWidgetPluginManager
   *   Field widget plugin manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   * @param \Drupal\canvas\ShapeMatcher\PropSourceSuggester $prop_source_suggester
   *   The prop source suggester service.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel.
   * @param \Drupal\canvas\PropShape\PropShapeRepositoryInterface $propShapeRepository
   *   The prop shape repository service.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    array $plugin_definition,
    ComponentValidator $componentValidator,
    WidgetPluginManager $fieldWidgetPluginManager,
    EntityTypeManagerInterface $entityTypeManager,
    PropSourceSuggester $prop_source_suggester,
    LoggerChannelInterface $logger,
    PropShapeRepositoryInterface $propShapeRepository,
  ) {
    assert(array_key_exists('local_source_id', $configuration));
    // Expect prop_field_definitions to be set by the config entity.
    assert(array_key_exists('prop_field_definitions', $configuration));

    parent::__construct(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $componentValidator,
      $fieldWidgetPluginManager,
      $entityTypeManager,
      $prop_source_suggester,
      $logger,
      $propShapeRepository,
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get(ComponentValidator::class),
      $container->get('plugin.manager.field.widget'),
      $container->get(EntityTypeManagerInterface::class),
      $container->get(PropSourceSuggester::class),
      $container->get('logger.channel.canvas'),
      $container->get(PropShapeRepositoryInterface::class),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return parent::defaultConfiguration() + [
    // Component name (duplicates source_local_id from entity)
      'local_source_id' => NULL,
      // The following will be populated from the Component config entity
      // settings.
    // SDC-style props schema.
      'props' => [],
    // SDC-style slots definition.
      'slots' => [],
    // Human-readable label.
      'label' => NULL,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isBroken(): bool {
    // External JavaScript components are always available.
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function getReferencedPluginClass(): ?string {
    // JavaScript components don't have PHP classes.
    return NULL;
  }

  /**
   * {@inheritdoc}
   *
   * Creates a mock SDC plugin for JavaScript components. This mock plugin
   * allows us to reuse the explicit UX field generation logic implemented by
   * the parent GeneratedFieldExplicitInputUxComponentSourceBase class.
   *
   * @todo Remove in https://www.drupal.org/project/canvas/issues/3503038
   */
  public function getSdcPlugin(): ComponentPlugin {
    if ($this->componentPlugin === NULL) {
      $this->componentPlugin = static::createSdcPluginFromConfiguration($this->configuration);
    }
    return $this->componentPlugin;
  }

  /**
   * {@inheritdoc}
   */
  protected function getComponentPlugin(): ComponentPlugin {
    return $this->getSdcPlugin();
  }

  /**
   * {@inheritdoc}
   */
  public function getComponentDescription(): TranslatableMarkup {
    $component = $this->getComponentPlugin();
    return new TranslatableMarkup('JavaScript component: %name', [
      '%name' => $component->metadata->name,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function renderComponent(array $inputs, array $slot_definitions, string $componentUuid, bool $isPreview): array {
    // Extract values from EvaluationResult objects.
    [$props, $props_cacheability] = self::getResolvedPropsAndCacheability($inputs[self::EXPLICIT_INPUT_NAME] ?? []);

    // JavaScript rendering variant.
    if (isset($this->configuration['javascript'])) {
      $build = [
        '#extjs_component_source' => $this,
        '#extjs_props' => $props,
        '#extjs_component_uuid' => $componentUuid,
        '#extjs_is_preview' => $isPreview,
        '#pre_render' => [[static::class, 'preRenderJavascriptVariant']],
      ];
      $props_cacheability->applyTo($build);
      return $build;
    }

    // Custom elements preview variant.
    $custom_element = CustomElement::create($this->configuration['local_source_id']);

    foreach ($props as $key => $value) {
      $custom_element->setAttribute($key, $value);
    }

    $build = [
      '#custom_element' => $custom_element,
      '#extjs_component_uuid' => $componentUuid,
      '#extjs_is_preview' => $isPreview,
      '#extjs_preview_config' => $this->configuration['custom_elements_preview'] ?? [],
      '#pre_render' => [[static::class, 'preRenderCustomElementsVariant']],
    ];
    $props_cacheability->applyTo($build);
    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function setSlots(array &$build, array $slots): void {
    // Store slot render arrays for later processing in #pre_render.
    $build['#slots'] = $slots;
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks(): array {
    return [
      'preRenderJavascriptVariant',
      'preRenderCustomElementsVariant',
    ];
  }

  /**
   * Adds Canvas slot marker comments to a slot render array.
   *
   * Wraps the slot render array with HTML comment markers using Markup objects
   * for #prefix and #suffix. Markup objects bypass XSS filtering, allowing
   * HTML comments to be preserved in the rendered output.
   *
   * @param array $slotRenderArray
   *   The slot content as a render array.
   * @param string $componentUuid
   *   The UUID of the component containing this slot.
   * @param string $slotName
   *   The name of the slot.
   *
   * @return array
   *   A render array with #prefix and #suffix containing slot marker comments.
   */
  private static function addSlotCommentsRenderArray(array $slotRenderArray, string $componentUuid, string $slotName): array {
    $slotStart = sprintf('<!-- canvas-slot-start-%s/%s -->', $componentUuid, $slotName);
    $slotEnd = sprintf('<!-- canvas-slot-end-%s/%s -->', $componentUuid, $slotName);
    // Wrap the HTML comments in markup objects to avoid XSS-filtering to
    // remove the comments!
    return [
      '#prefix' => Markup::create($slotStart),
      'content' => $slotRenderArray,
      '#suffix' => Markup::create($slotEnd),
    ];
  }

  /**
   * Pre-render callback for JavaScript rendering variant.
   *
   * @param array $element
   *   Render array with #extjs_component_source, #extjs_props, and #slots.
   *
   * @return array
   *   Render array with component container.
   */
  public static function preRenderJavascriptVariant(array $element): array {
    $componentSource = $element['#extjs_component_source'];
    $settings = $componentSource->configuration;
    $elementId = Html::getUniqueId('extjs-component');
    $componentUuid = $element['#extjs_component_uuid'] ?? NULL;
    $isPreview = $element['#extjs_is_preview'] ?? FALSE;

    $slotMarkup = [];
    if (!empty($element['#slots'])) {
      foreach ($element['#slots'] as $slotName => $slotContent) {
        // Add slot marker comments for preview mode.
        if ($isPreview && $componentUuid) {
          $slotContent = static::addSlotCommentsRenderArray($slotContent, $componentUuid, $slotName);
        }

        $slotMarkup[$slotName] = \Drupal::service('renderer')->renderInIsolation($slotContent);
      }
    }

    // Create the preview element.
    $element['preview'] = [
      '#type' => 'html_tag',
      '#tag' => 'div',
      '#attributes' => [
        'id' => $elementId,
        'class' => ['extjs-component-container'],
        'data-component-name' => $settings['local_source_id'],
        'data-component-props' => json_encode($element['#extjs_props']),
        'data-component-slots' => json_encode($slotMarkup),
        'data-component-bundles' => json_encode($settings['javascript']),
      ],
      '#attached' => [
        'library' => ['canvas_extjs/component_loader'],
      ],
    ];

    return $element;
  }

  /**
   * Pre-render callback for custom elements preview variant.
   *
   * @param array $element
   *   Render array with #custom_element, #extjs_preview_config, and #slots.
   *
   * @return array
   *   Render array with custom elements preview.
   */
  public static function preRenderCustomElementsVariant(array $element): array {
    /** @var \Drupal\custom_elements\CustomElement $custom_element */
    $custom_element = $element['#custom_element'];
    $componentUuid = $element['#extjs_component_uuid'] ?? NULL;
    $isPreview = $element['#extjs_is_preview'] ?? FALSE;
    $previewConfig = $element['#extjs_preview_config'] ?? [];

    if (!empty($element['#slots'])) {
      foreach ($element['#slots'] as $slot_name => $slot_content) {
        if (!is_array($slot_content)) {
          continue;
        }

        if (isset($slot_content['#custom_element'])) {
          $custom_element->setSlotFromCustomElement($slot_name, $slot_content['#custom_element']);
        }
        else {
          // Add slot marker comments for preview mode.
          if ($isPreview && $componentUuid) {
            $slot_content = static::addSlotCommentsRenderArray($slot_content, $componentUuid, $slot_name);
          }
          $custom_element->setSlotFromRenderArray($slot_name, $slot_content);
        }
      }
    }

    // Get the preview provider based on configuration.
    $providerPluginId = $previewConfig['preview_provider'] ?? 'auto';
    if ($providerPluginId === 'auto') {
      // Use auto-detection via resolver.
      $resolver = \Drupal::service('custom_elements.preview_resolver');
      $provider = $resolver->getProvider(\Drupal::request());
    }
    else {
      // Instantiate the specific provider.
      /** @var \Drupal\custom_elements\PreviewProvider\CustomElementsPreviewProviderManager $providerManager */
      $providerManager = \Drupal::service('plugin.manager.custom_elements_preview_provider');
      $provider = $providerManager->createInstance($providerPluginId);

      // Set base URL if configured.
      if (!empty($previewConfig['base_url']) && method_exists($provider, 'setBaseUrl')) {
        $provider->setBaseUrl($previewConfig['base_url']);
      }
    }

    $element['preview'] = $provider->preview($custom_element);
    return $element;
  }

  /**
   * Create test Component config entities for JavaScript components.
   *
   * This creates and saves test component entities. If a component name
   * is provided, only that component is created. Otherwise, multiple
   * test components may be created.
   *
   * @param string|null $component_name
   *   Optional component name. If provided, only creates that component.
   */
  public static function createTestComponentEntities(?string $component_name = NULL): void {
    $components = [
      'TestButton' => [
        'label' => 'Test Button',
        'category' => 'Vue Components',
        'props' => [
          'type' => 'object',
          'properties' => [
            'label' => [
              'type' => 'string',
              'title' => 'Button Label',
              'default' => 'Click me',
              'examples' => ['Submit', 'Cancel', 'Learn More'],
            ],
            'variant' => [
              'type' => 'string',
              'title' => 'Button Variant',
              'default' => 'primary',
              'enum' => ['primary', 'secondary', 'success', 'danger'],
              'examples' => ['primary', 'success'],
            ],
            'size' => [
              'type' => 'string',
              'title' => 'Button Size',
              'default' => 'medium',
              'enum' => ['small', 'medium', 'large'],
              'examples' => ['medium'],
            ],
            'disabled' => [
              'type' => 'boolean',
              'title' => 'Disabled',
              'default' => FALSE,
              'examples' => [FALSE],
            ],
          ],
        ],
      ],

      'TestCard' => [
        'label' => 'Test Card',
        'category' => 'Vue Components',
        'props' => [
          'type' => 'object',
          'required' => ['title'],
          'properties' => [
            'title' => [
              'type' => 'string',
              'title' => 'Card Title',
              'default' => 'Card Title',
              'examples' => ['Featured Article', 'Product Showcase'],
            ],
            'description' => [
              'type' => 'string',
              'title' => 'Card Description',
              'default' => 'Card description goes here',
              'examples' => ['Learn more about our services', 'Discover new features'],
            ],
          ],
        ],
      ],

      // TwoColumnLayout component with slots.
      'TwoColumnLayout' => [
        'label' => 'Two Column Layout',
        'category' => 'Vue Components',
        'props' => [
          'type' => 'object',
          'properties' => [
            'width' => [
              'type' => 'integer',
              'title' => 'First Column Width (%)',
              'description' => 'The width of the first column as a percentage',
              'default' => 50,
              'enum' => [25, 33, 34, 50, 66, 67, 75],
              'examples' => [33, 50, 66],
            ],
          ],
        ],
        'slots' => [
          'column-one' => [
            'title' => 'Column One',
            'description' => 'The contents of the first column.',
            'examples' => ['<p>This is column 1 content</p>'],
          ],
          'column-two' => [
            'title' => 'Column Two',
            'description' => 'The contents of the second column.',
            'examples' => ['<p>This is column 2 content</p>'],
          ],
        ],
      ],

      'TestPopover' => [
        'label' => 'Test Popover',
        'category' => 'Vue Components',
        'props' => [
          'type' => 'object',
          'properties' => [
            'placement' => [
              'type' => 'string',
              'title' => 'Popover Placement',
              'default' => 'bottom',
              'enum' => ['top', 'bottom', 'left', 'right', 'top-start', 'top-end', 'bottom-start', 'bottom-end'],
              'examples' => ['bottom', 'top'],
            ],
            'trigger' => [
              'type' => 'string',
              'title' => 'Trigger Type',
              'default' => 'click',
              'enum' => ['click', 'hover'],
              'examples' => ['click'],
            ],
            'offset' => [
              'type' => 'integer',
              'title' => 'Offset (px)',
              'default' => 8,
              'examples' => [8, 12, 16],
            ],
            'showArrow' => [
              'type' => 'boolean',
              'title' => 'Show Arrow',
              'default' => TRUE,
              'examples' => [TRUE],
            ],
            'closeOnClickOutside' => [
              'type' => 'boolean',
              'title' => 'Close on Click Outside',
              'default' => TRUE,
              'examples' => [TRUE],
            ],
            'delay' => [
              'type' => 'integer',
              'title' => 'Delay (ms)',
              'default' => 0,
              'examples' => [0, 200, 500],
            ],
          ],
        ],
      ],

      'TestMarkup' => [
        'label' => 'Test Markup',
        'category' => 'Vue Components',
        'props' => [
          'type' => 'object',
          'required' => ['content'],
          'properties' => [
            'content' => [
              'type' => 'string',
              'title' => 'HTML Content',
              'examples' => ['<p>Hello World</p>', '<div>Custom HTML content</div>'],
            ],
          ],
        ],
      ],
    ];

    // If a specific component is requested, filter to just that one.
    if ($component_name !== NULL) {
      if (!isset($components[$component_name])) {
        // If component not found, return early.
        return;
      }
      $components = [$component_name => $components[$component_name]];
    }

    foreach ($components as $name => $config) {
      $snake_case_name = static::convertToSnakeCase($name);
      $component_id = 'extjs.' . $snake_case_name;
      $existing = ComponentEntity::load($component_id);
      if ($existing) {
        continue;
      }

      // Create the component settings.
      $settings = [
        'label' => $config['label'],
        'local_source_id' => $name,
        'props' => $config['props'],
        'slots' => $config['slots'] ?? [],
        // Initialize empty, will be populated by ::ensurePropFieldDefinitions()
        'prop_field_definitions' => [],
      ];

      $component = ComponentEntity::create([
        'id' => $component_id,
        'label' => $config['label'],
        'category' => $config['category'],
        'source' => self::SOURCE_PLUGIN_ID,
        'provider' => NULL,
        'source_local_id' => $name,
        'active_version' => 'v1',
        'versioned_properties' => [
          VersionedConfigEntityBase::ACTIVE_VERSION => ['settings' => $settings],
        ],
        'status' => TRUE,
      ]);

      // Ensure prop field definitions are set up.
      static::ensurePropFieldDefinitions($component);
      $component->save();
    }
  }

  /**
   * Ensures all props have field definitions for a component entity.
   *
   * Generates missing field definitions from the props schema using SDC's
   * prop generation logic and updates the component entity's settings.
   *
   * @param \Drupal\canvas\Entity\ComponentInterface $component
   *   The component entity to update.
   */
  public static function ensurePropFieldDefinitions(ComponentInterface $component): void {
    // Load the active version to work with the settings.
    $version = $component->getActiveVersion();
    $component->loadVersion($version);

    $settings = $component->getSettings();

    // Initialize prop_field_definitions if not set.
    if (!isset($settings['prop_field_definitions'])) {
      $settings['prop_field_definitions'] = [];
      $component->setSettings($settings);
    }

    if (!empty($settings['props']['properties'])) {

      // Check if any prop is missing its field definition.
      $prop_names = array_keys($settings['props']['properties']);
      $missing_props = array_diff($prop_names, array_keys($settings['prop_field_definitions']));

      if (!empty($missing_props)) {
        // Create mock SDC plugin and generate field definitions.
        $sdc_plugin = static::createSdcPluginFromConfiguration([
          'local_source_id' => $component->get('source_local_id'),
          'label' => $settings['label'] ?? $component->label(),
          'props' => $settings['props'],
          'slots' => $settings['slots'] ?? [],
        ]);
        $generated_definitions = self::getPropsForComponentPlugin($sdc_plugin);

        foreach ($missing_props as $prop_name) {
          if (isset($generated_definitions[$prop_name])) {
            $settings['prop_field_definitions'][$prop_name] = $generated_definitions[$prop_name];
          }
        }
        $component->setSettings($settings);
      }
    }
  }

  /**
   * Creates a mock SDC plugin from configuration.
   *
   * This static helper creates a mock SDC plugin that can be used to generate
   * prop field definitions without needing to instantiate the full class.
   *
   * @param array $configuration
   *   Configuration array that must contain:
   *   - local_source_id: The component's local source ID.
   *   - label: The component's human-readable label.
   *   - props: SDC-style props schema array.
   *   - slots: SDC-style slots definition array.
   *
   * @return \Drupal\Core\Plugin\Component
   *   A mock SDC plugin configured with the given settings.
   */
  protected static function createSdcPluginFromConfiguration(array $configuration): ComponentPlugin {
    // The Component class expects to create metadata from plugin_definition.
    // We need to structure the plugin_definition to include all metadata.
    $plugin_definition = [
      'id' => 'extjs:' . $configuration['local_source_id'],
      'machineName' => $configuration['local_source_id'],
      'path' => 'mock-path',
      'name' => $configuration['label'],
      'template' => 'mock-template.html.twig',
      'provider' => NULL,
      // Direct SDC props structure.
      'props' => $configuration['props'],
      // Direct SDC slots structure.
      'slots' => $configuration['slots'],
    ];

    // Component class creates its own metadata from plugin_definition.
    return new SdcPlugin(
      ['app_root' => '', 'enforce_schemas' => FALSE],
      'extjs:' . $configuration['local_source_id'],
      $plugin_definition
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getSourceLabel(): TranslatableMarkup {
    return $this->t('JavaScript component');
  }

  /**
   * {@inheritdoc}
   */
  public function checkRequirements(): void {
    // JavaScript components requirements.
    if (empty($this->configuration['local_source_id'])) {
      throw new ComponentDoesNotMeetRequirementsException(['Component name is required']);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies(): array {
    $dependencies = parent::calculateDependencies();

    // Add custom_elements dependency if using custom_elements_preview.
    if (isset($this->configuration['custom_elements_preview'])) {
      $dependencies['module'][] = 'custom_elements';
    }

    return $dependencies;
  }

}
