# Canvas External JS

Provides External JavaScript component source for Canvas.

## Overview

This module adds a new component source plugin to Canvas that allows you
to integrate external JavaScript-rendered components (like Vue, React, or
Nuxt components) into Canvas pages.

The External JavaScript component source works by:
1. Storing component metadata (props, slots) within Canvas component
   configuration entities
2. Rendering components using custom JavaScript, either provided as custom
   JavaScript files or loaded via the Custom Elements previews

## Requirements

- Drupal 11.2+
- Canvas module
- Optional: Custom Elements module (required for Custom Elements preview
  variants and decoupled rendering)

## Installation

Install as you would normally install a contributed Drupal module.

## Use Cases

This module supports two primary use cases:

### 1. Client-Side Rendering within Drupal

Render JavaScript components client-side within Drupal pages:
- **Via Custom JavaScript**: Load custom JavaScript bundles that render
  components directly in the browser
- **Via Custom Elements Previews**: Use the Custom Elements module's
  pluggable preview system to render components (supports Nuxt, Astro,
  and other frameworks)

### 2. Fully Decoupled Rendering

Build fully decoupled frontends with server-side rendering:
- Canvas pages are provided via API (as Custom Elements markup or JSON)
- Frontend frameworks handle rendering with full server-side rendering
  support
- Requires: Custom Elements module + Lupus CE Renderer or Lupus Decoupled

## Integrations

- **[Custom Elements](https://www.drupal.org/project/custom_elements)**:
  Provides a pluggable component-preview system that may be used to render
  components. Currently supports Nuxt. Also supports processing Canvas
  pages into custom elements, either serialized as markup or JSON.
- **[nuxt-component-preview](https://github.com/drunomics/nuxt-component-preview)**:
  Nuxt module that exposes Vue components from Nuxt and generates a
  component-index compatible with this module.
- **[Lupus CE Renderer](https://www.drupal.org/project/lupus_ce_renderer)**
  or **[Lupus Decoupled](https://www.drupal.org/project/lupus_decoupled)**:
  Add these modules to get a full-page API for Custom Elements pages.
- **[Lupus Decoupled](https://www.drupal.org/project/lupus_decoupled)**:
  Builds upon the Custom Elements module integration to provide a
  fully-decoupled setup with comprehensive API endpoints.

See [TESTING.md](TESTING.md) for complete developer-oriented examples and
instructions on how to test them.

## Component Registration

Canvas External JS supports component registration from a component index
file.

### Component Index Format

Components are defined in a JSON file following the component-index schema,
which is basically the same as the Drupal Single Directory Components
schema.

- **Schema specification**:
  [schema/component-index.schema.json](schema/component-index.schema.json)
- **Example with 5 components**:
  [tests/fixtures/component-index.json](tests/fixtures/component-index.json)

### Rendering Variants

Components can be rendered using different methods:

**JavaScript Bundle**
- Loads external JavaScript bundles directly
- No custom_elements module required

**Custom Elements Preview with auto provider (default)**
- Uses Custom Elements module with automatic preview provider. When using
  Lupus Decoupled, this would pick up the configured preview provider.
- Requires: custom_elements module

**Custom Elements Preview with specific provider**
- Uses Custom Elements module with selected preview provider (e.g., nuxt)
- Requires: custom_elements module, specified base_url


### Drush Commands for registration

Register components from a component index:

```bash
# Default: simple custom_elements_preview with auto provider
drush canvas:extjs-register https://example.com/component-index.json

# With specific preview provider (requires --base-url)
drush canvas:extjs-register https://example.com/index.json \
  --custom-elements-preview-provider=nuxt \
  --base-url=http://localhost:3000

# With auto provider explicitly
drush canvas:extjs-register https://example.com/index.json \
  --custom-elements-preview-provider=auto

# With JavaScript bundle URLs
drush canvas:extjs-register https://example.com/index.json \
  --javascript=https://cdn.example.com/components.js
```

Unregister components from a source:

```bash
drush canvas:extjs-unregister https://example.com/component-index.json
drush canvas:extjs-unregister /path/to/component-index.json
```

### Programmatic Registration

```php
$manager = \Drupal::service('canvas_extjs.component_index_manager');

// Register with default (simple custom_elements_preview).
$result = $manager->register('https://example.com/component-index.json');

// Register with detailed custom_elements_preview.
$url = 'https://example.com/component-index.json';
$result = $manager->register($url, [
  'custom_elements_preview' => [
    'preview_provider' => 'nuxt',
    'base_url' => 'http://localhost:3000',
  ],
]);

// Register with JavaScript bundle URLs.
// Note: When multiple URLs provided, they are loaded in order and the
// render() function is invoked on the LAST module.
$result = $manager->register($url, [
  'javascript' => [
    'https://cdn.example.com/react.min.js',     // Dependency
    'https://cdn.example.com/components.js',    // Main bundle
  ],
]);

// Unregister components.
$result = $manager->unregister($url);
```

## Providing custom JavaScript bundles

When using the JavaScript rendering with custom bundles, the last
specified bundle / JavaScript file must export a `render()` function,
that is used for rendering components. The function gets props, slots,
as well as the target DOM element passed. For example:

```javascript
/**
 * Renders a test button component.
 *
 * @param {HTMLElement} container
 *   The render container element.
 * @param {Object} props
 *   Component prop values (e.g., {label: 'Click me', variant:
 *   'primary'}).
 * @param {Object.<string, string>} slots
 *   Component slots as HTML markup strings (e.g., {default:
 *   '<p>Content</p>'}).
 *
 * @return {HTMLElement}
 *   The container element.
 */
function renderTestButton(container, props, slots) {
  container.innerHTML = '';

  const button = document.createElement('button');
  const variant = props.variant || 'primary';
  button.className = `test-button test-button--${variant}`;
  button.textContent = props.label || 'Click me';
  container.appendChild(button);
  return container;
}

/**
 * Main render function.
 *
 * @param {HTMLElement} container
 *   The container element to render into.
 * @param {string} componentName
 *   The component name (PascalCase).
 * @param {Object} props
 *   Component prop values as key-value pairs.
 * @param {Object.<string, string>} slots
 *   Component slots as HTML markup strings.
 *
 * @return {HTMLElement|null}
 *   The container element on success, null if component not found.
 */
export async function render(container, componentName, props, slots) {
  const renderers = {
    renderTestButton,
    renderTwoColumnLayout,
    ...
  };
  const renderFn = renderers[`render${componentName}`];
  return renderFn?.(container, props, slots) ?? null;
}
```

**Key Points:**
- Component names are PascalCase (e.g., 'TestButton', 'TwoColumnLayout')
- Multiple bundles: Last bundle exports `render()`, earlier bundles can
  be dependencies
- Slots: Are passed as HTML strings to be rendered. Possibly containing
  HTML that triggers loading JavaScript components again.

## Automated Testing

### PHPUnit Tests

Run all kernel tests:

```bash
./vendor/bin/phpunit --testdox web/modules/contrib/canvas_extjs/tests/
```

### Code Standards (PHPCS)

Check coding standards:

```bash
./vendor/bin/phpcs --standard=web/modules/contrib/canvas_extjs/phpcs.xml \
  web/modules/contrib/canvas_extjs/
```

Fix automatically where possible:

```bash
./vendor/bin/phpcbf --standard=web/modules/contrib/canvas_extjs/phpcs.xml \
  web/modules/contrib/canvas_extjs/
```

### Static Analysis (PHPStan)

Run PHPStan analysis:

```bash
./vendor/bin/phpstan analyze \
  --configuration=web/modules/contrib/canvas_extjs/phpstan.neon \
  web/modules/contrib/canvas_extjs/
```

## Example Frontend

A complete Nuxt example demonstrating Canvas External JS integration is
available from the [DrupalCon Vienna 2025 session](https://github.com/balintbrews/drupalcon-vienna-2025-canvas-js-frontend/tree/main/nuxt-example).

## AI disclosure

AI is being used to support the development of the module. However, every change
is going through a full, human review process.

## Supporting Organizations

**[drunomics](https://drunomics.com)** - Development, Maintenance
