<?php

declare(strict_types=1);

namespace Drupal\Tests\canvas_extjs\Kernel;

use Drupal\canvas\Entity\Component as ComponentEntity;
use Drupal\canvas\Entity\VersionedConfigEntityBase;
use Drupal\canvas\PropExpressions\StructuredData\EvaluationResult;
use Drupal\canvas_extjs\Plugin\Canvas\ComponentSource\ExternalJavaScriptComponent;
use Drupal\KernelTests\KernelTestBase;

/**
 * Tests rendering variant configuration and registration.
 *
 * Focuses on variant configuration schemas, validation rules, registration
 * behavior, versioning, and dependency calculation.
 *
 * @see \Drupal\Tests\canvas_extjs\Kernel\ComponentRenderingTest
 *
 * @group canvas_extjs
 */
class RenderingVariantTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'canvas',
    'canvas_extjs',
    'custom_elements',
    'field',
    'text',
    'user',
    'options',
  ];

  /**
   * Test config schema for all rendering variants.
   */
  public function testConfigSchema(): void {
    // Test auto provider.
    $component = ComponentEntity::create([
      'id' => 'extjs.test_auto',
      'label' => 'Test Auto',
      'category' => 'Test',
      'source' => ExternalJavaScriptComponent::SOURCE_PLUGIN_ID,
      'source_local_id' => 'TestAuto',
      'active_version' => VersionedConfigEntityBase::ACTIVE_VERSION,
      'versioned_properties' => [
        VersionedConfigEntityBase::ACTIVE_VERSION => [
          'settings' => [
            'local_source_id' => 'TestAuto',
            'props' => ['type' => 'object', 'properties' => []],
            'slots' => [],
            'label' => 'Test Auto',
            'prop_field_definitions' => [],
            'custom_elements_preview' => ['preview_provider' => 'auto'],
          ],
        ],
      ],
    ]);
    $component->save();
    $loaded = ComponentEntity::load('extjs.test_auto');
    $loaded->loadVersion($loaded->getActiveVersion());
    $this->assertEquals('auto', $loaded->getSettings()['custom_elements_preview']['preview_provider']);

    // Test specific provider.
    $component = ComponentEntity::create([
      'id' => 'extjs.test_specific',
      'label' => 'Test Specific',
      'category' => 'Test',
      'source' => ExternalJavaScriptComponent::SOURCE_PLUGIN_ID,
      'source_local_id' => 'TestSpecific',
      'active_version' => VersionedConfigEntityBase::ACTIVE_VERSION,
      'versioned_properties' => [
        VersionedConfigEntityBase::ACTIVE_VERSION => [
          'settings' => [
            'local_source_id' => 'TestSpecific',
            'props' => ['type' => 'object', 'properties' => []],
            'slots' => [],
            'label' => 'Test Specific',
            'prop_field_definitions' => [],
            'custom_elements_preview' => [
              'preview_provider' => 'nuxt',
              'base_url' => 'http://localhost:3000',
            ],
          ],
        ],
      ],
    ]);
    $component->save();
    $loaded = ComponentEntity::load('extjs.test_specific');
    $loaded->loadVersion($loaded->getActiveVersion());
    $this->assertEquals('nuxt', $loaded->getSettings()['custom_elements_preview']['preview_provider']);
    $this->assertEquals('http://localhost:3000', $loaded->getSettings()['custom_elements_preview']['base_url']);

    // Test javascript.
    $component = ComponentEntity::create([
      'id' => 'extjs.test_js',
      'label' => 'Test JS',
      'category' => 'Test',
      'source' => ExternalJavaScriptComponent::SOURCE_PLUGIN_ID,
      'source_local_id' => 'TestJS',
      'active_version' => VersionedConfigEntityBase::ACTIVE_VERSION,
      'versioned_properties' => [
        VersionedConfigEntityBase::ACTIVE_VERSION => [
          'settings' => [
            'local_source_id' => 'TestJS',
            'props' => ['type' => 'object', 'properties' => []],
            'slots' => [],
            'label' => 'Test JS',
            'prop_field_definitions' => [],
            'javascript' => ['https://cdn.example.com/components.js'],
          ],
        ],
      ],
    ]);
    $component->save();
    $loaded = ComponentEntity::load('extjs.test_js');
    $loaded->loadVersion($loaded->getActiveVersion());
    $this->assertEquals(['https://cdn.example.com/components.js'], $loaded->getSettings()['javascript']);
  }

  /**
   * Test validation errors.
   */
  public function testValidation(): void {
    $manager = \Drupal::service('canvas_extjs.component_index_manager');
    $fixtureUrl = \Drupal::service('extension.list.module')->getPath('canvas_extjs') . '/tests/fixtures/component-index.json';

    // Multiple variants.
    try {
      $manager->register($fixtureUrl, [
        'custom_elements_preview' => ['preview_provider' => 'auto'],
        'javascript' => ['https://example.com/bundle.js'],
      ]);
      $this->fail('Expected InvalidArgumentException');
    }
    catch (\InvalidArgumentException $e) {
      $this->assertStringContainsString('Only one rendering variant allowed', $e->getMessage());
    }

    // Missing base_url for specific provider.
    try {
      $manager->register($fixtureUrl, [
        'custom_elements_preview' => ['preview_provider' => 'nuxt'],
      ]);
      $this->fail('Expected InvalidArgumentException');
    }
    catch (\InvalidArgumentException $e) {
      $this->assertStringContainsString('base_url is required', $e->getMessage());
    }
  }

  /**
   * Test register() with rendering variants.
   */
  public function testRegisterVariants(): void {
    $manager = \Drupal::service('canvas_extjs.component_index_manager');
    $fixtureUrl = \Drupal::service('extension.list.module')->getPath('canvas_extjs') . '/tests/fixtures/component-index.json';

    // Default: auto provider.
    $manager->register($fixtureUrl);
    $component = ComponentEntity::load('extjs.test_button');
    $component->loadVersion($component->getActiveVersion());
    $this->assertEquals('auto', $component->getSettings()['custom_elements_preview']['preview_provider']);
    $this->assertContains('custom_elements', $component->get('dependencies')['module'] ?? []);

    // Specific provider with base_url.
    $manager->register($fixtureUrl, [
      'custom_elements_preview' => [
        'preview_provider' => 'nuxt',
        'base_url' => 'http://localhost:3000',
      ],
    ]);
    $component = ComponentEntity::load('extjs.test_button');
    $component->loadVersion($component->getActiveVersion());
    $this->assertEquals('nuxt', $component->getSettings()['custom_elements_preview']['preview_provider']);
    $this->assertEquals('http://localhost:3000', $component->getSettings()['custom_elements_preview']['base_url']);
    $this->assertContains('custom_elements', $component->get('dependencies')['module'] ?? []);

    // JavaScript variant.
    $manager->register($fixtureUrl, [
      'javascript' => ['https://cdn.example.com/components.js'],
    ]);
    \Drupal::entityTypeManager()->getStorage('component')->resetCache(['extjs.test_button']);
    $component = ComponentEntity::load('extjs.test_button');
    $component->loadVersion($component->getActiveVersion());
    $settings = $component->getSettings();
    $this->assertEquals(['https://cdn.example.com/components.js'], $settings['javascript']);
    $this->assertArrayNotHasKey('custom_elements_preview', $settings);

    // Note: Dependencies are UNION of all versions.
    // Since we registered with custom_elements_preview first (creating v1),
    // then javascript (creating v2), the entity dependencies include both.
    // This is correct Canvas behavior - dependencies merged from all versions.
    $this->assertGreaterThan(1, count($component->getVersions()), 'Should have multiple versions after re-registration with different rendering config');
  }

  /**
   * Test component versioning when props change.
   */
  public function testComponentVersioning(): void {
    $manager = \Drupal::service('canvas_extjs.component_index_manager');
    $fixtureUrl = \Drupal::service('extension.list.module')->getPath('canvas_extjs') . '/tests/fixtures/component-index.json';

    // Register initial component.
    $manager->register($fixtureUrl, ['javascript' => ['./test.js']]);

    $component = ComponentEntity::load('extjs.test_button');
    $initialVersion = $component->getActiveVersion();
    $component->loadVersion($initialVersion);
    $initialProps = $component->getSettings()['props'];

    $this->assertArrayHasKey('label', $initialProps['properties']);
    $this->assertArrayHasKey('variant', $initialProps['properties']);
    $this->assertArrayHasKey('disabled', $initialProps['properties']);
    $this->assertCount(1, $component->getVersions(), 'Should have one version initially');

    // Re-register with SAME props (should not create new version).
    $manager->register($fixtureUrl, ['javascript' => ['./test.js']]);

    $component = ComponentEntity::load('extjs.test_button');
    $sameVersion = $component->getActiveVersion();

    $this->assertEquals($initialVersion, $sameVersion, 'Version hash should not change when props unchanged');
    $this->assertCount(1, $component->getVersions(), 'Should still have one version when unchanged');

    // Create modified fixture with changed props.
    $data = json_decode(file_get_contents($fixtureUrl), TRUE);
    unset($data['components'][0]['props']['properties']['disabled']);
    $data['components'][0]['props']['properties']['icon'] = [
      'type' => 'string',
      'title' => 'Icon',
    ];

    $modifiedFixture = sys_get_temp_dir() . '/modified-component-index.json';
    file_put_contents($modifiedFixture, json_encode($data));

    // Re-register with CHANGED props (should create new version).
    $manager->register($modifiedFixture, ['javascript' => ['./test.js']]);

    $component = ComponentEntity::load('extjs.test_button');
    $newVersion = $component->getActiveVersion();

    $this->assertNotEquals($initialVersion, $newVersion, 'Version hash should change when props change');
    $this->assertCount(2, $component->getVersions(), 'Should have two versions after props change');

    // Verify new version has new props.
    $component->loadVersion($newVersion);
    $newProps = $component->getSettings()['props'];

    $this->assertArrayHasKey('icon', $newProps['properties']);
    $this->assertArrayNotHasKey('disabled', $newProps['properties']);

    // Verify old version still has old props.
    $component->loadVersion($initialVersion);
    $oldProps = $component->getSettings()['props'];

    $this->assertArrayHasKey('disabled', $oldProps['properties']);
    $this->assertArrayNotHasKey('icon', $oldProps['properties']);
  }

  /**
   * Test javascript-only component has no custom_elements dependency.
   */
  public function testJavascriptOnlyDependencies(): void {
    $manager = \Drupal::service('canvas_extjs.component_index_manager');
    $fixtureUrl = \Drupal::service('extension.list.module')->getPath('canvas_extjs') . '/tests/fixtures/component-index.json';

    // Register ONLY with javascript (not custom_elements first).
    $manager->register($fixtureUrl, [
      'javascript' => ['./test.js'],
    ]);

    $component = ComponentEntity::load('extjs.test_card');
    $component->loadVersion($component->getActiveVersion());

    // Should NOT have custom_elements dependency.
    $deps = $component->get('dependencies');
    $this->assertNotContains('custom_elements', $deps['module'] ?? [], 'Javascript-only component should not depend on custom_elements');
  }

  /**
   * Test JavaScript rendering variant preview.
   */
  public function testJavascriptRendering(): void {
    $manager = \Drupal::service('canvas_extjs.component_index_manager');
    $fixtureUrl = \Drupal::service('extension.list.module')->getPath('canvas_extjs') . '/tests/fixtures/component-index.json';

    $manager->register($fixtureUrl, [
      'javascript' => ['/modules/contrib/canvas_extjs/tests/fixtures/test-components.js'],
    ]);

    // Test simple component.
    $component = ComponentEntity::load('extjs.test_button');
    $component->loadVersion($component->getActiveVersion());
    $source = $component->getComponentSource();
    $build = $source->renderComponent(
      ['props' => ['label' => new EvaluationResult('Test Label'), 'variant' => new EvaluationResult('primary')]],
      [],
      'test-uuid',
      TRUE
    );

    $this->assertEquals(['label' => 'Test Label', 'variant' => 'primary'], $build['#extjs_props']);
    $this->assertNotNull($build['#extjs_component_source']);

    $callback = $build['#pre_render'][0];
    $processed = call_user_func($callback, $build);
    $preview = $processed['preview'];

    $this->assertEquals('html_tag', $preview['#type']);
    $this->assertEquals('div', $preview['#tag']);
    $this->assertContains('extjs-component-container', $preview['#attributes']['class']);
    $this->assertEquals('TestButton', $preview['#attributes']['data-component-name']);
    $this->assertStringContainsString('Test Label', $preview['#attributes']['data-component-props']);
    $this->assertContains('canvas_extjs/component_loader', $preview['#attached']['library']);

    // Test nested components (layout with button in slot).
    $buttonBuild = $source->renderComponent(
      ['props' => ['label' => new EvaluationResult('Nested Button')]],
      [],
      'button-uuid',
      TRUE
    );

    $layoutComponent = ComponentEntity::load('extjs.two_column_layout');
    $layoutComponent->loadVersion($layoutComponent->getActiveVersion());
    /** @var \Drupal\canvas_extjs\Plugin\Canvas\ComponentSource\ExternalJavaScriptComponent $layoutSource */
    $layoutSource = $layoutComponent->getComponentSource();
    $layoutBuild = $layoutSource->renderComponent(
      ['props' => ['width' => new EvaluationResult(33)]],
      [],
      'layout-uuid',
      TRUE
    );
    $layoutSource->setSlots($layoutBuild, ['column-one' => $buttonBuild]);

    $callback = $layoutBuild['#pre_render'][0];
    $processed = call_user_func($callback, $layoutBuild);
    $preview = $processed['preview'];

    $this->assertEquals('TwoColumnLayout', $preview['#attributes']['data-component-name']);
    $this->assertStringContainsString('"width":33', $preview['#attributes']['data-component-props']);

    $slotData = json_decode($preview['#attributes']['data-component-slots'], TRUE);
    $this->assertArrayHasKey('column-one', $slotData);
    $this->assertStringContainsString('extjs-component-container', $slotData['column-one']);
    $this->assertStringContainsString('data-component-name="TestButton"', $slotData['column-one']);
    $this->assertStringContainsString('Nested Button', $slotData['column-one']);
  }

}
