<?php

namespace Drupal\captcha_protected_page\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides a CAPTCHA verification form for protected pages.
 */
class CaptchaForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'captcha_protected_page_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $request = $this->getRequest();
    $redirect_path = $request->query->get('redirect_path', '/');

    $form['captcha'] = [
      '#type' => 'captcha',
      '#captcha_type' => 'default',
    ];

    $form['redirect_path'] = [
      '#type' => 'hidden',
      '#value' => $redirect_path,
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Verify'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $redirect_url = $_SESSION['captcha_original_url'] ?? '/';
    $original_path = $_SESSION['captcha_original_path'] ?? '';

    // Clear session immediately.
    unset($_SESSION['captcha_original_url']);
    unset($_SESSION['captcha_original_path']);

    $this->setVerificationCookies($original_path);

    // Force immediate redirect.
    $response = new RedirectResponse($redirect_url);
    $response->send();
    exit;
  }

  /**
   * Sets verification cookies for matched protected paths.
   *
   * @param string $original_path
   *   The original requested path.
   */
  protected function setVerificationCookies($original_path) {
    $config = $this->config('captcha_protected_page.settings');
    $protected_paths = $this->getProtectedPaths($config->get('protected_paths'));
    $expire = time() + ($config->get('cookie_expiration') ?: 86400);
    $request = $this->getRequest();

    foreach ($protected_paths as $path) {
      if ($this->isPathMatch($original_path, $path)) {
        $this->setCookie($path, $expire, $request);
      }
    }
  }

  /**
   * Gets protected paths from configuration.
   *
   * @param string $paths_config
   *   The raw paths configuration string.
   *
   * @return array
   *   An array of trimmed protected paths.
   */
  protected function getProtectedPaths($paths_config) {
    $paths = $paths_config ?: '';
    return array_filter(array_map('trim', explode("\n", $paths)));
  }

  /**
   * Checks if the original path matches a protected path.
   *
   * @param string $original_path
   *   The original requested path.
   * @param string $protected_path
   *   A protected path to check against.
   *
   * @return bool
   *   TRUE if the path matches, FALSE otherwise.
   */
  protected function isPathMatch($original_path, $protected_path) {
    return !empty($protected_path) && strpos($original_path, $protected_path) === 0;
  }

  /**
   * Sets a verification cookie.
   *
   * @param string $path
   *   The protected path.
   * @param int $expire
   *   Cookie expiration timestamp.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   */
  protected function setCookie($path, $expire, Request $request) {
    $cookie_name = 'captcha_protected_' . md5($path);
    $host = $request->getHost();

    setcookie(
      $cookie_name,
      'verified',
      [
        'expires' => $expire,
        'path' => $path,
        'domain' => '.' . $host,
        'secure' => $request->isSecure(),
        'httponly' => TRUE,
        'samesite' => 'Lax',
      ]
    );

    $request->cookies->set($cookie_name, 'verified');

    $this->logger('captcha_protected_page')->notice('Set cookie: @name for path @path', [
      '@name' => $cookie_name,
      '@path' => $path,
    ]);
  }

}
