<?php

namespace Drupal\captcha_protected_page\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Cookie;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Component\Utility\Crypt;

/**
 * Provides a CAPTCHA verification form for protected pages.
 */
class CaptchaForm extends FormBase {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new CaptchaForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'captcha_protected_page_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $request = $this->getRequest();
    $redirect_path = $request->query->get('redirect_path', '/');

    $form['captcha'] = [
      '#type' => 'captcha',
      '#captcha_type' => 'default',
    ];

    $form['redirect_path'] = [
      '#type' => 'hidden',
      '#value' => $redirect_path,
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Verify'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $redirect_url = NULL;

    $request = $this->getRequest();
    $session = $request->getSession();
    if ($session->has('captcha_original_url')) {
      $redirect_url = $session->get('captcha_original_url');
      $original_path = $session->get('captcha_original_path', '');
      // Clear session values immediately after using.
      $session->remove('captcha_original_url');
      $session->remove('captcha_original_path');
    }
    else {
      // Fallback: no session → front page with warning.
      $this->messenger()->addWarning($this->t('Your session expired. You have been redirected to the front page.'));
      $redirect_url = Url::fromRoute('<front>')->toString();
      $original_path = '';
    }

    // Create response and add cookies if needed.
    $response = new RedirectResponse($redirect_url);

    if (!empty($original_path)) {
      $this->setVerificationCookies($original_path, $response);
    }

    // Redirect properly via Drupal response system.
    $form_state->setResponse($response);
  }

  /**
   * Sets verification cookies for matched protected paths.
   *
   * @param string $original_path
   *   The original requested path.
   * @param \Symfony\Component\HttpFoundation\RedirectResponse $response
   *   The redirect response to add cookies to.
   */
  protected function setVerificationCookies($original_path, RedirectResponse $response) {
    $config = $this->configFactory->get('captcha_protected_page.settings');
    $protected_paths = $this->getProtectedPaths($config->get('protected_paths'));
    $expire = time() + ($config->get('cookie_expiration') ?: 86400);
    $request = $this->getRequest();

    foreach ($protected_paths as $path) {
      if ($this->isPathMatch($original_path, $path)) {
        $cookie = $this->createCookie($path, $expire, $request);
        $response->headers->setCookie($cookie);
      }
    }
  }

  /**
   * Creates a verification cookie for a protected path.
   *
   * @param string $path
   *   The protected path.
   * @param int $expire
   *   Cookie expiration timestamp.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\Cookie
   *   The created cookie object.
   */
  protected function createCookie($path, $expire, Request $request) {
    // Use Drupal's Crypt::hashBase64().
    $cookie_name = 'captcha_protected_' . Crypt::hashBase64($path);

    // Normalize cookie path: if wildcard, trim to base directory.
    $cookie_path = $path;
    if (str_ends_with($path, '/*')) {
      $cookie_path = rtrim(substr($path, 0, -2), '/');
      if ($cookie_path === '') {
        $cookie_path = '/';
      }
    }

    $host = $request->getHost();

    return new Cookie(
      $cookie_name,
      'verified',
      $expire,
      $cookie_path,
      '.' . $host,
      $request->isSecure(),
    // Httponly.
      TRUE,
      FALSE,
      'Lax'
    );
  }

  /**
   * Gets protected paths from configuration.
   *
   * @param string $paths_config
   *   The raw paths configuration string.
   *
   * @return array
   *   An array of trimmed protected paths.
   */
  protected function getProtectedPaths($paths_config) {
    $paths = $paths_config ?: '';
    return array_filter(array_map('trim', explode("\n", $paths)));
  }

  /**
   * Checks if the original path matches a protected path.
   *
   * @param string $original_path
   *   The original requested path.
   * @param string $protected_path
   *   A protected path to check against.
   *
   * @return bool
   *   TRUE if the path matches, FALSE otherwise.
   */
  protected function isPathMatch($original_path, $protected_path) {
    $original_path = rtrim($original_path, '/');
    $protected_path = rtrim($protected_path, '/');

    // If wildcard is present, treat it as prefix match.
    if (str_ends_with($protected_path, '/*')) {
      $base = rtrim(substr($protected_path, 0, -2), '/');
      return strpos($original_path, $base) === 0;
    }

    return $original_path === $protected_path;
  }

}
