<?php

namespace Drupal\captcha_protected_page\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;
use Drupal\Component\Utility\Crypt;

/**
 * Subscriber for CAPTCHA-protected page redirection.
 */
class CaptchaRedirectSubscriber implements EventSubscriberInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The current route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Cached protected paths.
   *
   * @var array
   */
  protected $protectedPaths;

  /**
   * Constructs a new CaptchaRedirectSubscriber.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The current route match.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    AccountInterface $current_user,
    RouteMatchInterface $route_match,
    LoggerChannelFactoryInterface $logger_factory,
    RequestStack $request_stack
  ) {
    $this->configFactory = $config_factory;
    $this->currentUser = $current_user;
    $this->routeMatch = $route_match;
    $this->logger = $logger_factory->get('captcha_protected_page');
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('current_user'),
      $container->get('current_route_match'),
      $container->get('logger.factory'),
      $container->get('request_stack')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      KernelEvents::REQUEST => ['checkCaptchaRequirement', 28],
    ];
  }

  /**
   * Checks if CAPTCHA verification is required for the current request.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The request event.
   */
  public function checkCaptchaRequirement(RequestEvent $event) {
    $request = $event->getRequest();
    $current_path = $request->getPathInfo();

    if ($this->shouldSkipVerification($request, $current_path)) {
      return;
    }

    $protected_paths = $this->getProtectedPaths();
    foreach ($protected_paths as $path) {
      if ($this->isPathProtected($current_path, $path)) {
        $this->handleProtectedPath($request, $current_path, $event);
        return;
      }
    }
  }

  /**
   * Determines if verification should be skipped for the current request.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   * @param string $current_path
   *   The current request path.
   *
   * @return bool
   *   TRUE if verification should be skipped, FALSE otherwise.
   */
  protected function shouldSkipVerification(Request $request, string $current_path): bool {
    // Skip for authenticated users unless their roles are selected.
    if ($this->currentUser->isAuthenticated()) {
      $config = $this->configFactory->get('captcha_protected_page.settings');
      if (!$config) {
        return TRUE;
      }

      $enabled_roles = $config->get('roles') ?? [];
      $user_roles = $this->currentUser->getRoles();

      if (empty(array_intersect($user_roles, $enabled_roles))) {
        return TRUE;
      }
    }

    // Skip CAPTCHA page itself and form submissions.
    if ($this->routeMatch->getRouteName() === 'captcha_protected_page.captcha_form' ||
        $request->getMethod() === 'POST') {
      return TRUE;
    }

    // Additional safeguard: skip if we're already in a redirect loop.
    $requestUri = $request->getUri();
    if (strpos($requestUri, 'captcha_protected_page/captcha') !== FALSE) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * Gets the list of protected paths from configuration.
   *
   * @return array
   *   An array of protected paths.
   */
  protected function getProtectedPaths(): array {
    if (!isset($this->protectedPaths)) {
      $config = $this->configFactory->get('captcha_protected_page.settings');
      if (!$config) {
        $this->protectedPaths = [];
        return [];
      }

      $paths = $config->get('protected_paths');
      $this->protectedPaths = is_string($paths) ?
        array_filter(array_map('trim', explode("\n", $paths))) : [];
    }
    return $this->protectedPaths;
  }

  /**
   * Checks if the current path matches a protected path pattern.
   *
   * @param string $current_path
   *   The current request path.
   * @param string $protected_path
   *   The protected path pattern.
   *
   * @return bool
   *   TRUE if the path is protected, FALSE otherwise.
   */
  protected function isPathProtected(string $current_path, string $protected_path): bool {
    if (empty($protected_path)) {
      return FALSE;
    }

    $current_path = rtrim($current_path, '/');
    $protected_path = rtrim($protected_path, '/');

    // Handle wildcard.
    if (str_ends_with($protected_path, '/*')) {
      $base = rtrim(substr($protected_path, 0, -2), '/');
      return strpos($current_path, $base) === 0;
    }

    // Exact match.
    return $current_path === $protected_path;
  }

  /**
   * Handles redirection for protected paths.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   * @param string $current_path
   *   The current request path.
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The request event.
   */
  protected function handleProtectedPath(Request $request, string $current_path, RequestEvent $event): void {
    $protected_paths = $this->getProtectedPaths();

    foreach ($protected_paths as $protected_path) {
      if ($this->isPathProtected($current_path, $protected_path)) {
        // Use Drupal's Crypt::hashBase64().
        $cookie_name = 'captcha_protected_' . Crypt::hashBase64($protected_path);
        $cookie_value = $request->cookies->get($cookie_name);

        $this->logger->notice('Checking cookie: @name = @value for path @path', [
          '@name' => $cookie_name,
          '@value' => $cookie_value ?? 'NOT SET',
          '@path' => $current_path,
        ]);

        if (empty($cookie_value)) {
          // Use the request's session.
          $session = $request->getSession();
          $session->set('captcha_original_url', $request->getUri());
          $session->set('captcha_original_path', $current_path);

          $redirect_url = Url::fromRoute('captcha_protected_page.captcha_form', [], [
            'query' => ['redirect_path' => $current_path],
            'absolute' => TRUE,
          ]);

          $response = new RedirectResponse($redirect_url->toString());
          $event->setResponse($response);
        }

        return;
      }
    }
  }

}
