<?php

namespace Drupal\captcha_protected_page\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\RoleStorageInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure CAPTCHA Protected Page settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The role storage service.
   *
   * @var \Drupal\user\RoleStorageInterface
   */
  protected $roleStorage;

  /**
   * The logger factory service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('entity_type.manager')->getStorage('user_role'),
      $container->get('logger.factory')
    );
  }

  /**
   * Constructs a new SettingsForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\user\RoleStorageInterface $role_storage
   *   The role storage service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    RoleStorageInterface $role_storage,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    parent::__construct($config_factory);
    $this->roleStorage = $role_storage;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['captcha_protected_page.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'captcha_protected_page_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('captcha_protected_page.settings');

    $roles = $this->roleStorage->loadMultiple();
    unset($roles['anonymous']);

    $form['protected_paths'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Protected Paths'),
      '#description' => $this->t('Enter one path per line. Wildcards: /downloads/*'),
      '#default_value' => $config->get('protected_paths'),
      '#required' => TRUE,
    ];

    $form['cookie_expiration'] = [
      '#type' => 'number',
      '#title' => $this->t('Cookie Expiration (seconds)'),
      '#default_value' => $config->get('cookie_expiration') ?? 86400,
      '#required' => TRUE,
      '#min' => 1,
      '#description' => $this->t('Time in seconds before verification expires. Examples:<br>
• 300 = 5 minutes<br>
• 3600 = 1 hour<br>
• 86400 = 24 hours<br>
• 604800 = 1 week'),
    ];

    $form['roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Additional Roles requiring CAPTCHA'),
      '#description' => $this->t('Anonymous users always require CAPTCHA. Select additional roles that should also complete verification.'),
      '#options' => array_map(function ($role) {
        return $role->label();
      }, $roles),
      '#default_value' => $config->get('roles') ?: [],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('captcha_protected_page.settings')
      ->set('protected_paths', $form_state->getValue('protected_paths'))
      ->set('cookie_expiration', $form_state->getValue('cookie_expiration'))
      ->set('roles', array_filter($form_state->getValue('roles')))
      ->save();

    $this->loggerFactory->get('captcha_protected_page')->info('Configuration updated');
    parent::submitForm($form, $form_state);
  }

}
