<?php

namespace Drupal\captchafox\CaptchaFox;

use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Server-side validation of the CaptchaFox code.
 */
class CaptchaFox {

  use StringTranslationTrait;

  private const string CAPTCHAFOX_API = 'https://api.captchafox.com/siteverify';

  private const array CAPTCHAFOX_ERRORS = [
    'missing-input-secret' => 'The secret parameter is missing.',
    'invalid-input-secret' => 'The secret parameter is invalid or malformed.',
    'missing-input-response' => 'The response parameter is missing.',
    'invalid-input-response	' => 'The response parameter is invalid or malformed.',
    'bad-request' => 'The request is invalid or malformed.',
    'timeout-or-duplicate' => 'The response is no longer valid.',
  ];

  /**
   * Sets the site key.
   *
   * @var string siteKey
   */
  protected string $siteKey = '';

  /**
   * Sets the secret key.
   *
   * @var string secretKey
   */
  protected string $secretKey = '';

  /**
   * Sets the errors array.
   *
   * @var array errors
   */
  protected array $errors = [];

  /**
   * Sets the success flag.
   *
   * @var bool success
   */
  private bool $success = FALSE;

  /**
   * Sets the validated flag.
   *
   * @var mixed validated
   */
  private mixed $validated;

  /**
   * Sets the request method.
   *
   * @var RequestMethodInterface requestMethod
   */
  private RequestMethodInterface $requestMethod;

  /**
   * Constructor.
   */
  public function __construct($site_key, $secret_key, $attributes, RequestMethodInterface $requestMethod) {
    $this->siteKey = $site_key;
    $this->secretKey = $secret_key;
    $this->requestMethod = $requestMethod;

    if (!empty($attributes) && is_array($attributes)) {
      foreach ($attributes as $name => $attribute) {
        if (isset($this->attributes[$name])) {
          $this->attributes[$name] = $attribute;
        }
      }
    }
  }

  /**
   * Build the CaptchaFox validation mechanism.
   *
   * @param string $response_token
   *   The response token.
   *
   * @return void
   *   The return value.
   */
  public function validate(string $response_token): void {

    $query = [
      'secret' => $this->secretKey,
      'response' => $response_token,
    ];
    $this->validated = $this->requestMethod->submit(self::CAPTCHAFOX_API, array_filter($query));
    if (isset($this->validated->success) && $this->validated->success === TRUE) {
      // Verified!
      $this->success = TRUE;
    }
    else {
      $this->errors = $this->getResponseErrors();
    }
  }

  /**
   * Get the response errors.
   *
   * @return array
   *   The return value.
   */
  public function getResponseErrors(): array {
    $errors = [];
    if (isset($this->validated->{'error-codes'}) && is_array($this->validated->{'error-codes'})) {
      $errorCodes = self::CAPTCHAFOX_ERRORS;
      $errorCodesFromResponse = $this->validated->{'error-codes'};
      foreach ($errorCodesFromResponse as $code) {
        $handledCode = isset($errorCodes[$code]) ? $code : 'unknown-error';
        $errors[] = $errorCodes[$handledCode];
      }
    }
    return $errors;
  }

  /**
   * Return the success flag.
   *
   * @return bool
   *   The boolean.
   */
  public function isSuccess(): bool {
    return $this->success;
  }

  /**
   * Get the errors.
   *
   * @return array
   *   The return value.
   */
  public function getErrors(): array {
    return $this->errors;
  }

}
