<?php

declare(strict_types=1);

namespace Drupal\cas_mock_server;

use Drupal\cas\Model\SslCertificateVerification;
use Drupal\cas\Service\CasHelper;

/**
 * Provides compatibility between CAS module versions for SSL certification.
 *
 * In CAS version 2, SSL verification constants are defined in CasHelper.
 * In CAS version 3, SSL verification constants are defined in the
 * SslCertificateVerification enum.
 *
 * This helper abstracts the version differences and provides a unified API.
 */
class CasVersionHelper {

  /**
   * Gets the value representing the default CA verification mode.
   *
   * @return int
   *   The constant value representing default certificate verification.
   */
  public function getDefaultValue(): int {
    if (class_exists(SslCertificateVerification::class)) {
      return SslCertificateVerification::Default->value;
    }
    if (class_exists(CasHelper::class)) {
      // @phpstan-ignore-next-line
      return CasHelper::CA_DEFAULT;
    }

    throw new \RuntimeException('CAS Helper class or SslCertificateVerification not found.');
  }

  /**
   * Gets the value representing no CA verification mode.
   *
   * @return int
   *   The constant value representing disabled certificate verification.
   */
  public function getNoneValue(): int {
    if (class_exists(SslCertificateVerification::class)) {
      return SslCertificateVerification::None->value;
    }
    if (class_exists(CasHelper::class)) {
      // @phpstan-ignore-next-line
      return CasHelper::CA_NONE;
    }

    throw new \RuntimeException('CAS Helper class or SslCertificateVerification not found.');
  }

  /**
   * Gets the value representing the custom CA verification mode.
   *
   * @return int
   *   The constant value representing custom certificate verification.
   */
  public function getCustomValue(): int {
    if (class_exists(SslCertificateVerification::class)) {
      return SslCertificateVerification::Custom->value;
    }
    if (class_exists(CasHelper::class)) {
      // @phpstan-ignore-next-line
      return CasHelper::CA_CUSTOM;
    }

    throw new \RuntimeException('CAS Helper class or SslCertificateVerification not found.');
  }

}
