<?php

declare(strict_types=1);

namespace Drupal\Tests\cas_user_ban_vbo\FunctionalJavascript;

use Drupal\cas_user_ban\CasUserBanManagerInterface;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Tests\cas\Traits\CasTestTrait;
use Drupal\Tests\cas_user_ban\Traits\BannedUsersTrait;
use Drupal\Tests\cas_user_ban\Traits\RadioOptionsTrait;
use Drupal\user\Entity\User;

/**
 * Tests the user cancel action form.
 *
 * @group cas_user_ban_vbo
 */
final class UserCancelActionFormTest extends WebDriverTestBase {

  use CasTestTrait;
  use BannedUsersTrait;
  use RadioOptionsTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'cas_user_ban_vbo_test',
    'cas_mock_server',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Field text to ban users.
   *
   * @var string
   */
  protected const FIELD_TEXT = 'Prevent users with an associated CAS username from re-creating the account.';

  /**
   * Tests the user ban through user multiple cancel form.
   */
  public function testBan(): void {
    // Create multiple users, some of them linked to CAS.
    $non_cas_user = $this->createUser();
    $bruce = $this->createUser();
    $this->createCasUser(
      'brucewayne',
      'brucewayne@waynecorporation.com',
      'ImBatman',
      local_account: $bruce
    );
    $damian = $this->createUser();
    $this->createCasUser(
      'damianwayne',
      'damianwayne@waynecorporation.com',
      'ImBatmansSon',
      local_account: $damian
    );
    \Drupal::service(CasUserBanManagerInterface::class)->add('damianwayne');

    // Go to the multiple cancel form.
    $admin_user = $this->createUser(['administer users']);
    $this->drupalLogin($admin_user);
    $assert_session = $this->assertSession();

    // Set the users to be cancelled.
    $this->drupalGet('/people-test');
    // Select all users including current user and superuser.
    for ($i = 0; $i < 5; $i++) {
      $assert_session->fieldExists("views_bulk_operations_bulk_form[$i]")->check();
    }
    $assert_session->selectExists('Action')->selectOption('Cancel the selected user accounts');
    $assert_session->buttonExists('Apply to selected items')->press();

    $assert_session->pageTextContains('Configure "Cancel the selected user accounts" action applied to the selection');
    $assert_session->pageTextContains($non_cas_user->getAccountName());
    $assert_session->pageTextContains($bruce->getAccountName());
    $assert_session->pageTextContains($damian->getAccountName());
    $assert_session->pageTextContains($admin_user->getAccountName());
    $assert_session->pageTextContains('admin');

    $cancellation_methods_fieldset = $assert_session->elementExists('named', [
      'fieldset',
      'When cancelling these accounts',
    ]);
    $this->assertRadioOptions($cancellation_methods_fieldset, [
      'Disable the account and keep its content.',
      'Disable the account and unpublish its content.',
      'Delete the account and make its content belong to the Anonymous user.',
      'Delete the account and its content.',
    ]);

    // The first radio option is selected by default.
    $this->assertTrue($assert_session->fieldExists('Disable the account and keep its content.', $cancellation_methods_fieldset)->isChecked());
    // As a consequence, the ban checkbox is not displayed.
    $ban_field = $assert_session->fieldExists(self::FIELD_TEXT);
    $this->assertFalse($ban_field->isVisible());
    $this->assertFalse($ban_field->isChecked());

    // By default only two last options will allow to ban and display the field.
    $assert_session->fieldExists('Disable the account and keep its content.', $cancellation_methods_fieldset)->press();
    $this->assertFalse($ban_field->isVisible());
    $this->assertFalse($ban_field->isChecked());
    $assert_session->fieldExists('Delete the account and make its content belong to the Anonymous user. This action cannot be undone.', $cancellation_methods_fieldset)->press();
    $this->assertTrue($ban_field->isVisible());
    $this->assertTrue($ban_field->isChecked());
    $assert_session->fieldExists('Disable the account and unpublish its content.', $cancellation_methods_fieldset)->press();
    $this->assertFalse($ban_field->isVisible());
    $this->assertFalse($ban_field->isChecked());

    // When the admin user selects any cancel option that allows to ban the user
    // the checkbox is shown.
    $assert_session->fieldExists('Delete the account and its content.', $cancellation_methods_fieldset)->press();
    $this->assertTrue($ban_field->isVisible());
    $this->assertTrue($ban_field->isChecked());

    // Submitting the form with the checkbox enabled will ban the users.
    $assert_session->buttonExists('Apply')->press();
    $assert_session->statusMessageContains("Account {$non_cas_user->getAccountName()} has been deleted.", 'status');
    $assert_session->statusMessageContains("Account {$bruce->getAccountName()} has been deleted.", 'status');
    $assert_session->statusMessageContains("Account {$damian->getAccountName()} has been deleted.", 'status');
    $assert_session->statusMessageContains('The CAS username brucewayne has been banned.', 'status');
    $assert_session->statusMessageContains('The CAS username damianwayne is already banned.', 'warning');
    $this->assertEquals(
      [0, 1, $admin_user->id()],
      array_keys(User::loadMultiple())
    );
    $this->assertBannedUsernames(['brucewayne', 'damianwayne']);
  }

  /**
   * Tests that the option to ban is available even for non CAS users.
   */
  public function testBanNonCasUser(): void {
    $non_cas_user = $this->createUser();
    $admin_user = $this->createUser(['administer users']);
    $this->drupalLogin($admin_user);

    // Set the users to be cancelled.
    $this->drupalGet('/people-test');
    $assert_session = $this->assertSession();
    $assert_session->fieldExists('views_bulk_operations_bulk_form[1]')->check();
    $assert_session->selectExists('Action')->selectOption('Cancel the selected user accounts');
    $assert_session->buttonExists('Apply to selected items')->press();
    $assert_session->pageTextContains('Configure "Cancel the selected user accounts" action applied to the selection');
    $assert_session->pageTextContains($non_cas_user->getAccountName());
    $assert_session->fieldNotExists(self::FIELD_TEXT);
  }

  /**
   * Tests possibility of altering the allowed cancel methods.
   */
  public function testAlteredCancelMethods(): void {
    // Install the module that alters the methods: adds a new custom restricting
    // to only that one.
    \Drupal::service('module_installer')->install(['cas_user_ban_test']);

    // Create multiple users.
    $bruce = $this->createUser();
    $this->createCasUser(
      'brucewayne',
      'brucewayne@waynecorporation.com',
      'ImBatman',
      local_account: $bruce
    );
    $admin_user = $this->createUser(['administer users']);
    $this->drupalLogin($admin_user);

    // Set the users to be cancelled.
    $this->drupalGet('/people-test');
    $assert_session = $this->assertSession();
    $assert_session->fieldExists('views_bulk_operations_bulk_form[1]')->check();
    $assert_session->selectExists('Action')->selectOption('Cancel the selected user accounts');
    $assert_session->buttonExists('Apply to selected items')->press();
    $assert_session->pageTextContains('Configure "Cancel the selected user accounts" action applied to the selection');
    $assert_session->pageTextContains($bruce->getAccountName());
    $cancellation_methods_fieldset = $assert_session->elementExists('named', [
      'fieldset',
      'When cancelling these accounts',
    ]);
    $this->assertRadioOptions($cancellation_methods_fieldset, [
      'Disable the account and keep its content.',
      'Disable the account and unpublish its content.',
      'Delete the account and its content.',
      'Test cancel method.',
    ]);

    // Check that new method displays the option to ban the user.
    $assert_session->fieldExists('Test cancel method.', $cancellation_methods_fieldset)->press();
    $ban_field = $assert_session->fieldExists(self::FIELD_TEXT);
    $this->assertTrue($ban_field->isVisible());
    $this->assertTrue($ban_field->isChecked());

    // Other methods than previous don't allow to ban the user.
    $assert_session->fieldExists('Delete the account and its content.', $cancellation_methods_fieldset)->press();
    $this->assertFalse($ban_field->isVisible());
    $this->assertFalse($ban_field->isChecked());
  }

}
