<?php

declare(strict_types=1);

namespace Drupal\cas_user_ban;

use Drupal\cas_user_ban\Exception\BanNotFoundException;
use Drupal\cas_user_ban\Exception\ExistingBanException;
use Drupal\cas_user_ban\Exception\InvalidCasUsernameException;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Class to manage banned users.
 */
class CasUserBanManager implements CasUserBanManagerInterface {

  use AutowireTrait;

  public function __construct(
    protected readonly Connection $connection,
    protected readonly TimeInterface $time,
    #[Autowire('@logger.channel.cas_user_ban')]
    protected readonly LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function add(string $username): void {
    if ($username === '') {
      throw new InvalidCasUsernameException('Username is empty.');
    }

    if ($this->isBanned($username)) {
      throw new ExistingBanException(sprintf('The ban for the CAS username "%s" already exists.', $username));
    }

    $this->connection->insert('cas_user_ban')
      ->fields([
        'cas_username' => $username,
        'timestamp' => $this->time->getRequestTime(),
      ])->execute();

    $this->logger->notice('The ban for the CAS username %username has been added.', ['%username' => $username]);
  }

  /**
   * {@inheritdoc}
   */
  public function isBanned(string $username): bool {
    $query = $this->connection->select('cas_user_ban', 'b')
      ->fields('b', ['cas_username'])
      ->condition('b.cas_username', $username)
      ->range(0, 1);

    return $query->execute()->fetchField() !== FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function remove(string $username): void {
    if (!$this->isBanned($username)) {
      throw new BanNotFoundException(sprintf('The ban for the CAS username "%s" could not be found.', $username));
    }

    $this->connection->delete('cas_user_ban')
      ->condition('cas_username', $username)
      ->execute();

    $this->logger->notice('The ban for the CAS username %username has been removed.', ['%username' => $username]);
  }

}
