<?php

declare(strict_types=1);

namespace Drupal\cas_user_ban\Form;

use Drupal\cas\Service\CasUserManager;
use Drupal\cas_user_ban\CasUserBanManagerInterface;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Form\FormInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;

/**
 * Form to ban several users.
 */
final class BanUsersForm implements ContainerInjectionInterface, FormInterface {

  use AutowireTrait;
  use StringTranslationTrait;

  public function __construct(
    TranslationInterface $string_translation,
  ) {
    $this->setStringTranslation($string_translation);
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'cas_user_ban_ban_users';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['intro'] = [
      '#prefix' => '<p>',
      '#markup' => $this->t('Use this form to ban one or more CAS usernames, preventing them to create an account.'),
      '#suffix' => '</p>',
    ];
    $form['cas_usernames'] = [
      '#type' => 'textarea',
      '#title' => $this->t('CAS username(s)'),
      '#required' => TRUE,
      '#default_value' => '',
      '#description' => $this->t('Enter one username per line.'),
    ];
    $form['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Ban users'),
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $cas_usernames = trim($form_state->getValue('cas_usernames'));
    $cas_usernames = preg_split('/[\n\r|\r|\n]+/', $cas_usernames);

    $batch_builder = (new BatchBuilder())
      ->setTitle($this->t('CAS ban users'))
      ->setFinishCallback([$this::class, 'banUsersFinished'])
      ->setInitMessage($this->t('Banning CAS usernames...'))
      ->setProgressMessage($this->t('Completed step @current of @total.'));

    foreach ($cas_usernames as $cas_username) {
      $cas_username = trim($cas_username);
      if (empty($cas_username)) {
        continue;
      }

      $batch_builder->addOperation([$this::class, 'banUser'], [$cas_username]);
    }

    batch_set($batch_builder->toArray());
  }

  /**
   * Perform a single user ban batch operation.
   *
   * Callback for batch_set().
   *
   * @param string $cas_username
   *   The CAS username that will be banned.
   * @param array $context
   *   The batch context array, passed by reference.
   */
  public static function banUser(string $cas_username, array &$context): void {
    $cas_user_ban_manager = \Drupal::service(CasUserBanManagerInterface::class);
    $cas_user_manager = \Drupal::service(CasUserManager::class);

    if ($cas_user_ban_manager->isBanned($cas_username)) {
      $context['results']['messages']['already_exists'][] = $cas_username;
      return;
    }

    if ($cas_user_manager->getUidForCasUsername($cas_username) !== NULL) {
      $cas_user_ban_manager->add($cas_username);
      $context['results']['messages']['has_account'][] = $cas_username;
      return;
    }

    $cas_user_ban_manager->add($cas_username);
    $context['results']['messages']['banned'][] = $cas_username;
  }

  /**
   * Complete ban users batch process.
   *
   * @param bool $success
   *   Indicate that the batch API tasks were all completed successfully.
   * @param array $results
   *   An array of all the results that were updated in update_do_one().
   * @param array $operations
   *   A list of all the operations that had not been completed by the batch
   *   API.
   */
  public static function banUsersFinished(bool $success, array $results, array $operations): void {
    $messenger = \Drupal::messenger();

    if ($success !== TRUE) {
      // An error occurred.
      // $operations contains the operations that remained unprocessed.
      $error_operation = reset($operations);
      $messenger->addError(t('An error occurred while processing %error_operation with arguments: @arguments', [
        '%error_operation' => $error_operation[0],
        '@arguments' => print_r($error_operation[1], TRUE),
      ]));

      return;
    }

    if (!empty($results['messages']['already_exists'])) {
      $messenger->addWarning(t('The following CAS usernames are already banned: %usernames.', [
        '%usernames' => implode(', ', $results['messages']['already_exists']),
      ]));
    }
    if (!empty($results['messages']['has_account'])) {
      $messenger->addWarning(t('The following CAS usernames were banned but an account is linked to them: %usernames.', [
        '%usernames' => implode(', ', $results['messages']['has_account']),
      ]));
    }
    if (!empty($results['messages']['banned'])) {
      $messenger->addStatus(t('The following CAS usernames were banned: %usernames.', [
        '%usernames' => implode(', ', $results['messages']['banned']),
      ]));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // No specific validation for this form.
  }

}
