<?php

declare(strict_types=1);

namespace Drupal\cas_user_ban\Traits;

use Drupal\cas_user_ban\Event\FilterUserCancelMethodEvent;
use Drupal\cas_user_ban\Exception\ExistingBanException;
use Drupal\Core\Render\Element;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Trait with helper functions to manage forms to cancel user accounts.
 */
trait UserCancelFormsTrait {

  use StringTranslationTrait;

  /**
   * The allowed cancel methods.
   *
   * @var list<string>
   */
  protected static array $allowedCancelMethods = [
    'user_cancel_reassign',
    'user_cancel_delete',
  ];

  /**
   * Adds a ban option to a form given a list of user IDs.
   *
   * The method also triggers the event needed to allow others to modify allowed
   * cancel methods and stores the values in form_state.
   *
   * @param array $form
   *   The build form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param list<string|int> $uids
   *   The list of user IDs to ban.
   * @param array|null $submit_element
   *   The submit element where to place the ban submit.
   */
  protected function addBanField(array &$form, FormStateInterface $form_state, array $uids, ?array &$submit_element = NULL): void {
    if (empty($uids)) {
      return;
    }

    $cas_usernames = [];
    foreach ($uids as $uid) {
      // Users should not be able to ban themselves.
      if ($this->currentUser->getAccount()->id() == $uid) {
        continue;
      }
      // Only users with CAS Username should be allowed to be banned.
      $cas_username = $this->casUserManager->getCasUsernameForAccount((int) $uid);
      if ($cas_username === FALSE) {
        continue;
      }

      $cas_usernames[] = $cas_username;
    }

    if (empty($cas_usernames)) {
      return;
    }
    // To make the ban work independently of the submit execution, like user
    // deletion before the ban, we store the usernames to ban.
    $form_state->set('cas_usernames', $cas_usernames);

    $cancel_methods = Element::getVisibleChildren($form['user_cancel_method']);
    $allowed_methods = $this->getCancelMethodsWithBanOption();

    $allowed_methods = array_intersect($allowed_methods, $cancel_methods);
    if (empty($allowed_methods)) {
      return;
    }

    $form_state->set('cas_ban_allowed_methods', $allowed_methods);
    $state_conditions = $this->buildOrValueCondition($allowed_methods);
    $cancel_method = $form_state->getValue('user_cancel_method', $form['user_cancel_method']['#default_value']);
    // @todo When banning users the option 'Require email confirmation' should
    //   not be allowed.
    $form['ban_cas_usernames'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Prevent users with an associated CAS username from re-creating the account.'),
      '#description' => $this->t("The username will be added to the banned list of CAS users and they won't be able to re-create the account."),
      '#default_value' => in_array($cancel_method, $allowed_methods),
      '#states' => [
        'visible' => [
          ':input[name="user_cancel_method"]' => $state_conditions,
        ],
        'checked' => [
          ':input[name="user_cancel_method"]' => $state_conditions,
        ],
      ],
    ];

    if ($submit_element !== NULL) {
      $submit_element[] = [$this, 'banSubmit'];
    }
  }

  /**
   * Gets user cancel methods for which a ban option should be shown.
   *
   * @return list<string>
   *   Machine names of applicable user cancel methods.
   */
  protected function getCancelMethodsWithBanOption(): array {
    $event = new FilterUserCancelMethodEvent(static::$allowedCancelMethods);
    $this->eventDispatcher->dispatch($event, FilterUserCancelMethodEvent::NAME);

    return $event->getAllowedMethods();
  }

  /**
   * Builds a condition with OR operator to use in '#states'.
   *
   * @param list<mixed> $values
   *   Values for which the condition should be true.
   *
   * @return list<'or'|array{value: mixed}>
   *   Condition array to use in states.
   */
  protected function buildOrValueCondition(array $values): array {
    $conditions = [];
    foreach ($values as $index => $value) {
      $conditions[] = ['value' => $value];
      if ($index !== array_key_last($values)) {
        $conditions[] = 'or';
      }
    }

    return $conditions;
  }

  /**
   * Submit to ban users based in the CAS usernames.
   *
   * @param array $form
   *   The build form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function banSubmit(array $form, FormStateInterface $form_state): void {
    $to_be_banned = $form_state->getValue('ban_cas_usernames');
    $cancel_method = $form_state->getValue('user_cancel_method');
    $cas_usernames = $form_state->get('cas_usernames');
    $allowed_methods = $form_state->get('cas_ban_allowed_methods', []);

    // Conditions to match in order to ban the user:
    // - Users are marked to be banned.
    // - Cancel method is part of allowed methods.
    if (!$to_be_banned || !in_array($cancel_method, $allowed_methods)) {
      return;
    }
    foreach ($cas_usernames as $cas_username) {
      $this->banUser($cas_username);
    }
  }

  /**
   * Bans the user displaying a message depending on the result.
   *
   * @param string $cas_username
   *   The CAS username to ban.
   */
  protected function banUser(string $cas_username): void {
    try {
      $this->casUserBanManager->add($cas_username);
      $this->messenger->addStatus($this->t('The CAS username %username has been banned.', ['%username' => $cas_username]));
    }
    catch (ExistingBanException) {
      $this->messenger->addWarning($this->t('The CAS username %username is already banned.', ['%username' => $cas_username]));
    }
  }

}
