<?php

declare(strict_types=1);

namespace Drupal\Tests\cas_user_ban\FunctionalJavascript;

use Drupal\cas_user_ban\CasUserBanManagerInterface;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Tests\cas\Traits\CasTestTrait;
use Drupal\Tests\cas_user_ban\Traits\RadioOptionsTrait;
use Drupal\user\Entity\User;

/**
 * Tests the user cancel form.
 *
 * @group cas_user_ban
 */
final class UserCancelFormTest extends WebDriverTestBase {

  use CasTestTrait;
  use RadioOptionsTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'cas_user_ban',
    'cas_mock_server',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Field text to ban users.
   *
   * @var string
   */
  protected const FIELD_TEXT = 'Prevent user from re-creating the account.';

  /**
   * Tests the user ban through user cancel form.
   */
  public function testBan(): void {
    // Create a user and link with CAS user.
    $user = $this->createUser();
    $this->createCasUser(
      'brucewayne',
      'brucewayne@waynecorporation.com',
      'ImBatman',
      local_account: $user
    );
    $cas_user_ban_manager = \Drupal::service(CasUserBanManagerInterface::class);
    $this->assertFalse($cas_user_ban_manager->isBanned('brucewayne'));

    // Go to the cancel form.
    $assert_session = $this->assertSession();
    $admin_user = $this->createUser(['administer users']);
    $this->drupalLogin($admin_user);
    $this->drupalGet("/user/{$user->id()}/cancel");
    $assert_session->pageTextContains("Are you sure you want to cancel the account {$user->getAccountName()}?");
    $cancellation_methods_fieldset = $assert_session->elementExists('named', ['fieldset', 'Cancellation method']);
    $this->assertRadioOptions($cancellation_methods_fieldset, [
      'Disable the account and keep its content.',
      'Disable the account and unpublish its content.',
      'Delete the account and make its content belong to the Anonymous user.',
      'Delete the account and its content.',
    ]);

    // The first radio option is selected by default.
    $this->assertTrue($assert_session->fieldExists('Disable the account and keep its content.', $cancellation_methods_fieldset)->isChecked());
    // As a consequence, the ban checkbox is not displayed.
    $ban_field = $assert_session->fieldExists(self::FIELD_TEXT);
    $this->assertFalse($ban_field->isVisible());
    $this->assertFalse($ban_field->isChecked());

    // When the admin user selects any cancel option that allows to ban the user
    // the checkbox is shown.
    // By default only the last two options will allow to ban and display the
    // field.
    $assert_session->fieldExists('Delete the account and make its content belong to the Anonymous user. This action cannot be undone.', $cancellation_methods_fieldset)->press();
    $this->assertTrue($ban_field->isVisible());
    $this->assertTrue($ban_field->isChecked());
    $assert_session->fieldExists('Disable the account and unpublish its content.', $cancellation_methods_fieldset)->press();
    $this->assertFalse($ban_field->isVisible());
    $this->assertFalse($ban_field->isChecked());
    $assert_session->fieldExists('Delete the account and its content.', $cancellation_methods_fieldset)->press();
    $this->assertTrue($ban_field->isVisible());
    $this->assertTrue($ban_field->isChecked());

    // Submitting the form with the checkbox enabled will display a message and
    // ban the user.
    $assert_session->buttonExists('Confirm')->press();
    $assert_session->statusMessageContains("Account {$user->getAccountName()} has been deleted.", 'status');
    $assert_session->statusMessageContains('The CAS username brucewayne has been banned.', 'status');
    $this->assertTrue($cas_user_ban_manager->isBanned('brucewayne'));
    $this->assertEquals([0, 1, $admin_user->id()], array_keys(User::loadMultiple()));

    // Already banned users will display a warning message.
    $another_user = $this->createUser();
    // Use the same CAS username.
    $this->createCasUser(
      'brucewayne',
      'brucewayne@waynecorporation.com',
      'ImStillBatman',
      local_account: $another_user
    );

    $this->drupalGet("/user/{$another_user->id()}/cancel");
    $assert_session->pageTextContains("Are you sure you want to cancel the account {$another_user->getAccountName()}?");
    $assert_session->fieldExists('Delete the account and make its content belong to the Anonymous user.', $cancellation_methods_fieldset)->press();
    $this->assertTrue($ban_field->isVisible());
    $this->assertTrue($ban_field->isChecked());

    $assert_session->buttonExists('Confirm')->press();
    $assert_session->statusMessageContains("Account {$another_user->getAccountName()} has been deleted.", 'status');
    $assert_session->statusMessageContains('The CAS username brucewayne is already banned.', 'warning');
    $this->assertTrue($cas_user_ban_manager->isBanned('brucewayne'));
    $this->assertEquals([0, 1, 3], array_keys(User::loadMultiple()));
  }

  /**
   * Tests that the option to ban is not available for non CAS users.
   */
  public function testBanNonCasUser(): void {
    $user = $this->createUser();
    $assert_session = $this->assertSession();
    $admin_user = $this->createUser(['administer users']);

    $this->drupalLogin($admin_user);
    $this->drupalGet("/user/{$user->id()}/cancel");

    $assert_session->pageTextContains("Are you sure you want to cancel the account {$user->getAccountName()}?");
    $assert_session->fieldNotExists(self::FIELD_TEXT);
  }

  /**
   * Tests that the option to ban is not available for own cancellation.
   */
  public function testSelfBan(): void {
    $assert_session = $this->assertSession();
    $user = $this->createUser(['cancel account']);
    $this->createCasUser(
      'brucewayne',
      'brucewayne@waynecorporation.com',
      'ImBatman',
      local_account: $user
    );

    $this->drupalLogin($user);
    $this->drupalGet("/user/{$user->id()}/cancel");

    $assert_session->pageTextContains("Are you sure you want to cancel your account?");
    $assert_session->fieldNotExists(self::FIELD_TEXT);
  }

  /**
   * Tests possibility of altering the allowed cancel methods.
   */
  public function testAlteredCancelMethods(): void {
    // Install the module that alters the methods: adds a new custom restricting
    // to only that one.
    \Drupal::service('module_installer')->install(['cas_user_ban_test']);

    // Create a user and link with CAS user.
    $user = $this->createUser();
    $this->createCasUser(
      'brucewayne',
      'brucewayne@waynecorporation.com',
      'ImBatman',
      local_account: $user
    );
    $cas_user_ban_manager = \Drupal::service(CasUserBanManagerInterface::class);
    $this->assertFalse($cas_user_ban_manager->isBanned('brucewayne'));

    // Go to the cancel form to verify the methods.
    $assert_session = $this->assertSession();
    $admin_user = $this->createUser(['administer users']);
    $this->drupalLogin($admin_user);
    $this->drupalGet("/user/{$user->id()}/cancel");
    $assert_session->pageTextContains("Are you sure you want to cancel the account {$user->getAccountName()}?");
    $cancellation_methods_fieldset = $assert_session->elementExists('named', ['fieldset', 'Cancellation method']);
    $this->assertRadioOptions($cancellation_methods_fieldset, [
      'Disable the account and keep its content.',
      'Disable the account and unpublish its content.',
      'Delete the account and its content.',
      'Test cancel method.',
    ]);

    // Check that new method displays the option to ban the user.
    $assert_session->fieldExists('Test cancel method.', $cancellation_methods_fieldset)->press();
    $ban_field = $assert_session->fieldExists(self::FIELD_TEXT);
    $this->assertTrue($ban_field->isVisible());
    $this->assertTrue($ban_field->isChecked());

    // Other methods than previous don't allow to ban the user.
    $assert_session->fieldExists('Delete the account and its content.', $cancellation_methods_fieldset)->press();
    $this->assertFalse($ban_field->isVisible());
    $this->assertFalse($ban_field->isChecked());
  }

}
