<?php

declare(strict_types=1);

namespace Drupal\cas_user_ban\Controller;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\PagerSelectExtender;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\Url;

/**
 * Displays the banned users list.
 */
class BannedUsersListController implements ContainerInjectionInterface {

  use AutowireTrait;
  use StringTranslationTrait;

  public function __construct(
    protected readonly Connection $connection,
    protected readonly DateFormatterInterface $dateFormatter,
    TranslationInterface $stringTranslation,
  ) {
    $this->stringTranslation = $stringTranslation;
  }

  /**
   * Builds the response.
   */
  public function __invoke(): array {
    $build['banned_users_list'] = [
      '#theme' => 'table',
      '#header' => [
        'cas_username' => [
          'data' => $this->t('CAS username'),
        ],
        'timestamp' => [
          'data' => $this->t('Banned on'),
        ],
        'operations' => [
          'data' => $this->t('Operations'),
        ],
      ],
      '#empty' => $this->t('No CAS usernames banned.'),
    ];

    $query = $this->connection->select('cas_user_ban', 'b')
      ->fields('b', ['cas_username', 'timestamp'])
      ->orderBy('b.timestamp', 'DESC')
      ->extend(PagerSelectExtender::class)
      ->limit(20);

    $banned_users = $query->execute()->fetchAll();

    foreach ($banned_users as $banned_user) {
      $links = [];
      $links['delete'] = [
        'title' => $this->t('Delete'),
        'url' => Url::fromRoute(
          'cas_user_ban.remove_user_ban',
          ['cas_username' => $banned_user->cas_username],
        ),
        'attributes' => [
          'class' => ['use-ajax'],
          'data-dialog-type' => 'modal',
          'data-dialog-options' => Json::encode([
            'width' => 880,
          ]),
        ],
      ];

      $build['banned_users_list']['#rows'][] = [
        'cas_username' => [
          'data' => $banned_user->cas_username,
        ],
        'timestamp' => [
          'data' => $this->dateFormatter->format($banned_user->timestamp, 'medium'),
        ],
        'operations' => [
          'data' => [
            '#type' => 'operations',
            '#links' => $links,
          ],
        ],
      ];
    }

    $build[] = ['#type' => 'pager'];
    $build['#attached']['library'][] = 'core/drupal.dialog.ajax';

    return $build;
  }

}
