<?php

declare(strict_types=1);

namespace Drupal\cas_user_ban\EventSubscriber;

use Drupal\cas\Event\CasPreRegisterEvent;
use Drupal\cas_user_ban\CasUserBanManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber for CasPreRegisterSubscriber.
 */
final class CasPreRegisterSubscriber implements EventSubscriberInterface {

  use StringTranslationTrait;

  public function __construct(
    protected readonly CasUserBanManagerInterface $casUserBanManager,
    #[Autowire('@logger.channel.cas_user_ban')]
    protected readonly LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      CasPreRegisterEvent::class => 'onPreRegister',
    ];
  }

  /**
   * Subscribe to the preregister event on user login.
   *
   * @param \Drupal\cas\Event\CasPreRegisterEvent $event
   *   The event subscribed.
   */
  public function onPreRegister(CasPreRegisterEvent $event): void {
    $username = $event->getDrupalUsername();

    // Prevent the register if the user is banned.
    if (!$this->casUserBanManager->isBanned($username)) {
      return;
    }

    $event->cancelAutomaticRegistration(
      $this->t('There was a problem logging in. Please contact a site administrator.'),
    );

    $this->logger->warning(
      'User registration attempt for banned user with CAS Username "%username".', ['%username' => $username]
    );
  }

}
