<?php

declare(strict_types=1);

namespace Drupal\cas_user_ban\Form;

use Drupal\cas_user_ban\CasUserBanManagerInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\Url;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Form to confirm ban deletion.
 */
final class RemoveBanForm extends ConfirmFormBase {

  use AutowireTrait;

  /**
   * The banned CAS username.
   *
   * @var string
   */
  protected $bannedCasUsername;

  public function __construct(
    protected CasUserBanManagerInterface $casUserBanManager,
    MessengerInterface $messenger,
    TranslationInterface $string_translation,
  ) {
    $this->setMessenger($messenger);
    $this->setStringTranslation($string_translation);
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'cas_user_ban_remove_user_ban';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t('Are you sure you want to unblock CAS username %banned_cas_username?', ['%banned_cas_username' => $this->bannedCasUsername]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('cas_user_ban.banned_users_list');
  }

  /**
   * {@inheritdoc}
   *
   * @param array $form
   *   A nested array form elements comprising the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param string $cas_username
   *   The CAS username to unban.
   */
  public function buildForm(array $form, FormStateInterface $form_state, $cas_username = '') {
    if (!$this->casUserBanManager->isBanned($cas_username)) {
      throw new NotFoundHttpException();
    }

    $this->bannedCasUsername = $cas_username;

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->casUserBanManager->remove($this->bannedCasUsername);
    // Display a message to the user since the manager already adds a log entry.
    $this->messenger()->addStatus($this->t('The ban for the CAS username %username has been deleted.', ['%username' => $this->bannedCasUsername]));
    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
