<?php

declare(strict_types=1);

namespace Drupal\cas_user_ban\Hook;

use Drupal\cas_user_ban\CasUserBanManagerInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Link;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;

/**
 * Hook implementations that add validation to the 'bulk_add_cas_users' form.
 *
 * @see \Drupal\cas\Form\BulkAddCasUsers
 */
final class BulkAddCasUsersHooks {

  use StringTranslationTrait;
  use DependencySerializationTrait;

  public function __construct(
    protected CasUserBanManagerInterface $casUserBanManager,
    TranslationInterface $string_translation,
  ) {
    $this->setStringTranslation($string_translation);
  }

  /**
   * Implements hook_form_FORM_ID_alter().
   *
   * @param array $form
   *   The form to be altered.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  #[Hook('form_bulk_add_cas_users_alter')]
  public function alter(array &$form, FormStateInterface $form_state): void {
    $form['cas_usernames']['#element_validate'][] = [$this, 'validateBannedUsers'];
  }

  /**
   * Validates that the field does not contain banned usernames.
   *
   * @param array $element
   *   The form element whose value is being validated.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function validateBannedUsers(array &$element, FormStateInterface $form_state): void {
    $cas_usernames = $form_state->getValue('cas_usernames');

    if ($cas_usernames === NULL) {
      return;
    }

    $invalid_usernames = [];
    $cas_usernames = preg_split('/[\n\r|\r|\n]+/', trim($cas_usernames));

    foreach ($cas_usernames as $cas_username) {
      $cas_username = trim($cas_username);
      if (!empty($cas_username) && $this->casUserBanManager->isBanned($cas_username)) {
        $invalid_usernames[] = $cas_username;
      }
    }

    if (empty($invalid_usernames)) {
      return;
    }

    $form_state->setError(
      $element,
      $this->t(
        'The %field contains invalid values, the following users are banned: %banned_users.<br>You can administer banned users in %link.',
        [
          '%field' => $element['#title'],
          '%banned_users' => implode(', ', $invalid_usernames),
          '%link' => Link::createFromRoute(
            $this->t('CAS banned users'),
            'cas_user_ban.banned_users_list',
            options: ['attributes' => ['target' => '_blank']],
          )->toString(),
        ],
      ),
    );
  }

}
