<?php

declare(strict_types=1);

namespace Drupal\Tests\cas_user_ban\Functional;

use Drupal\cas_user_ban\CasUserBanManagerInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Url;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests the banned users list page.
 *
 * @group cas_user_ban
 */
class BannedUsersListTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block',
    'cas_user_ban',
    'datetime_testing',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests list page.
   */
  public function testListPage(): void {
    $this->drupalPlaceBlock('local_tasks_block');
    $assert_session = $this->assertSession();

    // User without permission can't access the page.
    $this->drupalGet('/admin/people/cas/banned-users-list');
    $assert_session->statusCodeEquals(403);

    // User with permission can access.
    $this->drupalLogin($this->createUser(['administer users']));
    $this->drupalGet('/admin/people/cas/banned-users-list');
    $assert_session->statusCodeEquals(200);

    // Test that a link is displayed in users list page.
    $this->drupalGet('/admin/people');
    $this->clickLink('CAS banned users');
    $assert_session->addressEquals('/admin/people/cas/banned-users-list');

    // No banned users message.
    $assert_session->pageTextContains('No CAS usernames banned.');

    // Prepare predictable expected data.
    $cas_user_ban_manager = \Drupal::service(CasUserBanManagerInterface::class);
    $banned_users = [];
    for ($i = 0; $i < 30; $i++) {
      $cas_username = $i . '_' . $this->randomMachineName();
      \Drupal::time()->setTime('+1 day');
      $cas_user_ban_manager->add($cas_username);
      $banned_users[$cas_username] = \Drupal::time()->getRequestTime();
    }

    // Elements are ordered by timestamp descending.
    $banned_users = array_reverse($banned_users);

    // Pager limits the table to 10 elements.
    $this->drupalGet('/admin/people/cas/banned-users-list');
    $this->assertBannedUsersTable(array_slice($banned_users, 0, 20));

    // Visiting second page will show next items.
    $this->clickLink('Page 2');
    $this->assertBannedUsersTable(array_slice($banned_users, 20, 10));

    // Removing the ban to the users will reduce the list.
    $unban_users = array_slice($banned_users, 0, 25);
    array_walk($unban_users, fn ($time, $username) => $cas_user_ban_manager->remove($username));

    $this->drupalGet('/admin/people/cas/banned-users-list');
    $this->assertBannedUsersTable(array_diff($banned_users, $unban_users));
  }

  /**
   * Asserts the banned users listing table.
   *
   * @param array $expected_users
   *   Associative array containing username as key and timestamp as value.
   */
  protected function assertBannedUsersTable(array $expected_users): void {
    $assert_session = $this->assertSession();
    $table = $assert_session->elementExists('css', 'table');

    // Check that the table contains the expected header.
    $table_header = $assert_session->elementExists('css', 'thead', $table);
    $rows = $table_header->findAll('css', 'tr');
    $cols = $rows[0]->findAll('css', 'th');
    $this->assertEquals('CAS username', $cols[0]->getText());
    $this->assertEquals('Banned on', $cols[1]->getText());
    $this->assertEquals('Operations', $cols[2]->getText());

    // Check that table body matches expected values and order.
    $table_body = $assert_session->elementExists('css', 'tbody', $table);
    $rows = $table_body->findAll('css', 'tr');
    $this->assertCount(count($expected_users), $rows);
    $i = 0;
    foreach ($expected_users as $username => $timestamp) {
      $cols = $rows[$i]->findAll('css', 'td');
      // CAS username column.
      $this->assertEquals($username, $cols[0]->getText());
      // Banned on column.
      $expected_date = \Drupal::service(DateFormatterInterface::class)->format($timestamp, 'medium');
      $this->assertEquals($expected_date, $cols[1]->getText());
      // Operations column.
      $assert_session->elementExists('xpath', $assert_session->buildXPathQuery('//a[@href=:href][.=:text]', [
        ':href' => Url::fromRoute('cas_user_ban.remove_user_ban', ['cas_username' => $username])->toString(),
        ':text' => 'Delete',
      ]), $cols[2]);
      $i++;
    }
  }

}
