<?php

declare(strict_types=1);

namespace Drupal\Tests\cas_user_ban\Functional;

use Drupal\cas_user_ban\CasUserBanManagerInterface;
use Drupal\Core\Url;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\cas\Traits\CasTestTrait;
use Drupal\Tests\cas_user_ban\Traits\BannedUsersTrait;

/**
 * Tests the remove ban confirmation form.
 *
 * @group cas_user_ban
 */
class RemoveBanFormTest extends BrowserTestBase {

  use CasTestTrait;
  use BannedUsersTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'cas_user_ban',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests the form to remove a ban.
   */
  public function testRemoveBanForm(): void {
    // Ban some usernames to test the remove ban form.
    $cas_user_ban_manager = \Drupal::service(CasUserBanManagerInterface::class);
    $cas_user_ban_manager->add('the_penguin');
    $cas_user_ban_manager->add('bane');
    $cas_user_ban_manager->add('the_joker');

    $assert_session = $this->assertSession();

    // Users without permission can't access the route.
    $this->drupalGet('/admin/people/cas/remove-user-ban/the_penguin');
    $assert_session->statusCodeEquals(403);

    // Visit the form to remove one ban.
    $this->drupalLogin($this->createUser(['administer users']));
    $this->drupalGet('/admin/people/cas/remove-user-ban/the_penguin');
    $assert_session->pageTextContains('Are you sure you want to unblock CAS username the_penguin?');
    $assert_session->pageTextContains('This action cannot be undone.');
    $this->submitForm([], 'Confirm');

    // Check that username has been removed and there is a redirection, message
    // and log entry.
    $assert_session->addressEquals(Url::fromRoute('cas_user_ban.banned_users_list'));
    $assert_session->statusMessageContains('The ban for the CAS username the_penguin has been deleted.', 'status');
    $this->assertBannedUsernames(['bane', 'the_joker']);

    // Visit the form and remove another ban.
    $this->drupalGet('/admin/people/cas/remove-user-ban/the_joker');
    $assert_session->pageTextContains('Are you sure you want to unblock CAS username the_joker?');
    $assert_session->pageTextContains('This action cannot be undone.');
    $this->submitForm([], 'Confirm');

    // Visit the form and cancel operation.
    $assert_session->addressEquals(Url::fromRoute('cas_user_ban.banned_users_list'));
    $assert_session->statusMessageContains('The ban for the CAS username the_joker has been deleted.', 'status');
    $this->assertBannedUsernames(['bane']);

    // Check that cancel button does not remove any ban.
    $this->drupalGet('/admin/people/cas/remove-user-ban/bane');
    $assert_session->pageTextContains('Are you sure you want to unblock CAS username bane?');
    $assert_session->pageTextContains('This action cannot be undone.');
    $this->clickLink('Cancel');

    $assert_session->addressEquals(Url::fromRoute('cas_user_ban.banned_users_list'));
    $assert_session->statusMessageNotExists('status');
    $this->assertBannedUsernames(['bane']);

    // Visit the form with a non-banned user.
    $this->drupalGet('cas_user_ban.remove_user_ban', ['cas_username' => 'batman']);
    $assert_session->statusCodeEquals(404);
  }

}
