<?php

declare(strict_types=1);

namespace Drupal\Tests\cas_user_ban\Functional;

use Drupal\cas_user_ban\CasUserBanManagerInterface;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\cas\Traits\CasTestTrait;
use Drupal\user\Entity\User;
use Drupal\user\UserInterface;
use Psr\Log\LogLevel;

/**
 * Tests that banned users can not recreate their accounts.
 *
 * @group cas_user_ban
 */
class UserRegistrationTest extends BrowserTestBase {

  use CasTestTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block',
    'cas_mock_server',
    'cas_user_ban_test',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests registration.
   */
  public function testRegistrationPrevention(): void {
    // Place the login/logout block so that we can check if user is logged in.
    $this->placeBlock('system_menu_block:account');

    // Allow user registration on log in.
    $this->config('cas.settings')
      ->set('user_accounts.auto_register', TRUE)
      ->set('user_accounts.email_assignment_strategy', 1)
      ->set('user_accounts.email_attribute', 'email')
      ->save();

    $cas_user_ban_manager = \Drupal::service(CasUserBanManagerInterface::class);

    // Banned users won't be able to recreate their accounts when login in.
    // Add a CAS user without account linked.
    $this->createCasUser(
      'brucewayne',
      'brucewayne@waynecorporation.com',
      'ImBatman',
    );
    $this->assertCount(2, User::loadMultiple());

    $cas_user_ban_manager->add('brucewayne');
    $this->casLogin('brucewayne@waynecorporation.com', 'ImBatman');
    $assert_session = $this->assertSession();
    $assert_session->statusMessageContains('There was a problem logging in. Please contact a site administrator.', 'error');
    $this->assertCount(2, User::loadMultiple());
    $this->assertUserNotLoggedIn();

    $log_messages = \Drupal::state()->get('cas_user_ban_test.log_messages', []);
    $this->assertCount(1, $log_messages[LogLevel::WARNING]);
    $this->assertEquals(
      'User registration attempt for banned user with CAS Username "brucewayne".',
      $log_messages[LogLevel::WARNING][0],
    );

    // Other users not in the list will be created as usual and logged in.
    $this->createCasUser(
      'damianwayne',
      'damianwayne@waynecorporation.com',
      'ImBatmansSon',
    );
    $this->casLogin('damianwayne@waynecorporation.com', 'ImBatmansSon');
    $this->assertCount(3, User::loadMultiple());
    $assert_session->statusMessageNotExists('error');
    $account = user_load_by_name('damianwayne');
    $this->assertInstanceOf(UserInterface::class, $account);
    $this->assertUserLoggedIn();

    $log_messages = \Drupal::state()->get('cas_user_ban_test.log_messages', []);
    $this->assertCount(1, $log_messages[LogLevel::WARNING]);

    // Removing the ban allows the user to be created.
    $this->drupalLogout();
    $cas_user_ban_manager->remove('brucewayne');
    $this->casLogin('brucewayne@waynecorporation.com', 'ImBatman');
    $this->assertCount(4, User::loadMultiple());
    $assert_session->statusMessageNotExists('error');
    $account = user_load_by_name('brucewayne');
    $this->assertInstanceOf(UserInterface::class, $account);
    $this->assertUserLoggedIn();

    $log_messages = \Drupal::state()->get('cas_user_ban_test.log_messages', []);
    $this->assertCount(1, $log_messages[LogLevel::WARNING]);

    // Banning user does not prevent logging in.
    $this->drupalLogout();
    $cas_user_ban_manager->add('brucewayne');
    $this->casLogin('brucewayne@waynecorporation.com', 'ImBatman');
    $this->assertCount(4, User::loadMultiple());
    $assert_session->statusMessageNotExists('error');
    $this->assertUserLoggedIn();

    $log_messages = \Drupal::state()->get('cas_user_ban_test.log_messages', []);
    $this->assertCount(1, $log_messages[LogLevel::WARNING]);
  }

}
