<?php

declare(strict_types=1);

namespace Drupal\Tests\cas_user_ban\Functional;

use Drupal\cas_user_ban\CasUserBanManagerInterface;
use Drupal\Core\Url;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\cas\Traits\CasTestTrait;
use Drupal\user\Entity\User;

/**
 * Tests validation on bulk import CAS users.
 *
 * @group cas_user_ban
 */
class ValidateCasBulkAddTest extends BrowserTestBase {

  use CasTestTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'cas_mock_server',
    'cas_user_ban',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests the bulk import form.
   */
  public function testValidationBulkAddForm(): void {
    $this->createCasUser(
      'brucewayne',
      'brucewayne@waynecorporation.com',
      'ImBatman',
    );
    $this->createCasUser(
      'damianwayne',
      'damianwaynene@waynecorporation.com',
      'ImBatmansSon',
    );
    $this->createCasUser(
      'acethebathound',
      'acethebathound@waynecorporation.com',
      'ImBatmansDog',
    );

    $cas_user_ban_manager = \Drupal::service(CasUserBanManagerInterface::class);
    $cas_user_ban_manager->add('damianwayne');
    $cas_user_ban_manager->add('brucewayne');

    $this->drupalLogin($this->drupalCreateUser(['administer users']));
    $this->assertCount(3, User::loadMultiple());

    $this->drupalGet('/admin/people/create/cas-bulk');
    $assert_session = $this->assertSession();
    $this->submitForm([
      // Validation will remove extra spaces.
      'cas_usernames' => " damianwayne  \n \n \n\n brucewayne \n AceTheBatHound",
      'email_hostname' => 'sample.com',
      'roles[authenticated]' => TRUE,
    ], 'Create new accounts');

    // Validation of banned user will prevent the import of users displaying a
    // error message.
    $this->assertCount(3, User::loadMultiple());
    $assert_session->statusMessageContains(
      "The CAS username(s) contains invalid values, the following users are banned: damianwayne, brucewayne.You can administer banned users in CAS banned users.",
      'error',
    );
    $this->clickLink('CAS banned users');
    $assert_session->addressEquals(Url::fromRoute('cas_user_ban.banned_users_list'));

    // Check that removing one ban is not enough since there are two invalid
    // usernames.
    $cas_user_ban_manager->remove('damianwayne');
    $this->drupalGet('/admin/people/create/cas-bulk');
    $this->submitForm([
      'cas_usernames' => " damianwayne  \n \n \n\n brucewayne \n AceTheBatHound",
      'email_hostname' => 'sample.com',
      'roles[authenticated]' => TRUE,
    ], 'Create new accounts');
    $this->assertCount(3, User::loadMultiple());
    $assert_session->statusMessageContains(
      "The CAS username(s) contains invalid values, the following users are banned: brucewayne.You can administer banned users in CAS banned users.",
      'error',
    );
    $this->clickLink('CAS banned users');
    $assert_session->addressEquals(Url::fromRoute('cas_user_ban.banned_users_list'));

    // Removing the ban for all the invalid users will allow to import them.
    $cas_user_ban_manager->remove('brucewayne');
    $this->drupalGet('/admin/people/create/cas-bulk');
    $this->submitForm([
      'cas_usernames' => " damianwayne  \n \n \n\n brucewayne \n AceTheBatHound",
      'email_hostname' => 'sample.com',
      'roles[authenticated]' => TRUE,
    ], 'Create new accounts');
    $this->assertCount(6, User::loadMultiple());
    $assert_session->statusMessageNotExists('error');
  }

}
