<?php

declare(strict_types=1);

namespace Drupal\Tests\cas_user_ban\FunctionalJavascript;

use Drupal\cas_user_ban\CasUserBanManagerInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Tests\cas\Traits\CasTestTrait;
use Drupal\Tests\cas_user_ban\Traits\BannedUsersTrait;
use Drupal\Tests\cas_user_ban\Traits\RadioOptionsTrait;
use Drupal\user\Entity\User;

/**
 * Tests the multiple user cancel form.
 *
 * @group cas_user_ban
 */
final class UserMultipleCancelFormTest extends WebDriverTestBase {

  use CasTestTrait;
  use BannedUsersTrait;
  use RadioOptionsTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'cas_user_ban',
    'cas_mock_server',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Field text to ban users.
   *
   * @var string
   */
  protected const FIELD_TEXT = 'Prevent users with an associated CAS username from re-creating the account.';

  /**
   * Tests the user ban through user multiple cancel form.
   */
  public function testBan(): void {
    // Create multiple users, some of them linked to CAS.
    $non_cas_user = $this->createUser();
    $bruce = $this->createUser();
    $this->createCasUser(
      'brucewayne',
      'brucewayne@waynecorporation.com',
      'ImBatman',
      local_account: $bruce
    );
    $damian = $this->createUser();
    $this->createCasUser(
      'damianwayne',
      'damianwayne@waynecorporation.com',
      'ImBatmansSon',
      local_account: $damian
    );
    \Drupal::service(CasUserBanManagerInterface::class)->add('damianwayne');

    // Go to the multiple cancel form.
    $admin_user = $this->createUser(['administer users']);
    $this->drupalLogin($admin_user);
    // Set the users to be cancelled.
    /* @see \Drupal\user\Plugin\Action\CancelUser::executeMultiple() */
    $this->container->get(PrivateTempStoreFactory::class)
      ->get('user_user_operations_cancel')
      ->set($admin_user->id(), [
        $non_cas_user,
        $bruce,
        $damian,
        // We add current user too to check that the user does not get banned.
        $admin_user,
      ]);

    $assert_session = $this->assertSession();
    $this->drupalGet('/admin/people/cancel');
    $assert_session->pageTextContains('Are you sure you want to cancel these user accounts?');
    $cancellation_methods_fieldset = $assert_session->elementExists('named', ['fieldset', 'Cancellation method']);
    $this->assertRadioOptions($cancellation_methods_fieldset, [
      'Disable the account and keep its content.',
      'Disable the account and unpublish its content.',
      'Delete the account and make its content belong to the Anonymous user.',
      'Delete the account and its content.',
    ]);

    // The first radio option is selected by default.
    $this->assertTrue($assert_session->fieldExists('Disable the account and keep its content.', $cancellation_methods_fieldset)->isChecked());
    // As a consequence, the ban checkbox is not displayed.
    $ban_field = $assert_session->fieldExists(self::FIELD_TEXT);
    $this->assertFalse($ban_field->isVisible());
    $this->assertFalse($ban_field->isChecked());

    // When the admin user selects any cancel option that allows to ban the user
    // the checkbox is shown.
    // By default only the last two options will allow to ban and display the
    // field.
    $assert_session->fieldExists('Delete the account and make its content belong to the Anonymous user. This action cannot be undone.', $cancellation_methods_fieldset)->press();
    $this->assertTrue($ban_field->isVisible());
    $this->assertTrue($ban_field->isChecked());
    $assert_session->fieldExists('Disable the account and unpublish its content.', $cancellation_methods_fieldset)->press();
    $this->assertFalse($ban_field->isVisible());
    $this->assertFalse($ban_field->isChecked());
    $assert_session->fieldExists('Delete the account and its content.', $cancellation_methods_fieldset)->press();
    $this->assertTrue($ban_field->isVisible());
    $this->assertTrue($ban_field->isChecked());

    // Submitting the form with the checkbox enabled will ban the users.
    $assert_session->buttonExists('Confirm')->press();
    $assert_session->statusMessageContains("Account {$non_cas_user->getAccountName()} has been deleted.", 'status');
    $assert_session->statusMessageContains("Account {$bruce->getAccountName()} has been deleted.", 'status');
    $assert_session->statusMessageContains("Account {$damian->getAccountName()} has been deleted.", 'status');
    $assert_session->statusMessageContains('The CAS username brucewayne has been banned.', 'status');
    $assert_session->statusMessageContains('The CAS username damianwayne is already banned.', 'warning');
    $this->assertBannedUsernames(['brucewayne', 'damianwayne']);
    $this->assertEquals([0, 1, $admin_user->id()], array_keys(User::loadMultiple()));
  }

  /**
   * Tests that the option to ban is not available for non CAS users.
   */
  public function testBanNonCasUser(): void {
    $non_cas_user = $this->createUser();
    $admin_user = $this->createUser(['administer users']);
    $this->drupalLogin($admin_user);
    // Set the users to be cancelled.
    $this->container->get(PrivateTempStoreFactory::class)
      ->get('user_user_operations_cancel')
      ->set($admin_user->id(), [
        $non_cas_user,
      ]);

    $assert_session = $this->assertSession();
    $this->drupalGet('/admin/people/cancel');
    $assert_session->pageTextContains('Are you sure you want to cancel these user accounts?');
    $assert_session->fieldNotExists(self::FIELD_TEXT);
  }

  /**
   * Tests possibility of altering the allowed cancel methods.
   */
  public function testAlteredCancelMethods(): void {
    // Install the module that alters the methods: adds a new custom restricting
    // to only that one.
    \Drupal::service('module_installer')->install(['cas_user_ban_test']);

    // Create multiple users.
    $bruce = $this->createUser();
    $this->createCasUser(
      'brucewayne',
      'brucewayne@waynecorporation.com',
      'ImBatman',
      local_account: $bruce
    );
    $assert_session = $this->assertSession();
    $admin_user = $this->createUser(['administer users']);
    $this->drupalLogin($admin_user);
    $this->drupalGet('/admin/people');

    // Set the users to be cancelled.
    $this->container->get(PrivateTempStoreFactory::class)
      ->get('user_user_operations_cancel')
      ->set($admin_user->id(), [
        $bruce,
      ]);

    $this->drupalGet('/admin/people/cancel');

    $assert_session->pageTextContains('Are you sure you want to cancel these user accounts?');
    $cancellation_methods_fieldset = $assert_session->elementExists('named', ['fieldset', 'Cancellation method']);
    $this->assertRadioOptions($cancellation_methods_fieldset, [
      'Disable the account and keep its content.',
      'Disable the account and unpublish its content.',
      'Delete the account and its content.',
      'Test cancel method.',
    ]);

    // Check that new method displays the option to ban the user.
    $assert_session->fieldExists('Test cancel method.', $cancellation_methods_fieldset)->press();
    $ban_field = $assert_session->fieldExists(self::FIELD_TEXT);
    $this->assertTrue($ban_field->isVisible());
    $this->assertTrue($ban_field->isChecked());

    // Other methods than previous don't allow to ban the user.
    $assert_session->fieldExists('Delete the account and its content.', $cancellation_methods_fieldset)->press();
    $this->assertFalse($ban_field->isVisible());
    $this->assertFalse($ban_field->isChecked());
  }

}
