<?php

declare(strict_types=1);

namespace Drupal\Tests\cas_user_ban\Kernel;

use Drupal\cas_user_ban\CasUserBanManagerInterface;
use Drupal\cas_user_ban\Exception\BanNotFoundException;
use Drupal\cas_user_ban\Exception\ExistingBanException;
use Drupal\cas_user_ban\Exception\InvalidCasUsernameException;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Logger\LoggerChannel;
use Drupal\Core\Logger\LoggerChannelFactory;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\cas_user_ban\Traits\BannedUsersTrait;
use PHPUnit\Framework\MockObject\MockObject;
use Psr\Log\LogLevel;

/**
 * Tests CAS user ban manager service.
 *
 * @group cas_user_ban
 */
final class CasUserBanManagerTest extends KernelTestBase {

  use BannedUsersTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'externalauth',
    'cas',
    'cas_user_ban',
    'cas_user_ban_test',
  ];

  /**
   * The mocked logger factory.
   */
  protected LoggerChannelFactory|MockObject $loggerFactory;

  /**
   * The mocked log channel.
   */
  protected LoggerChannel|MockObject $loggerChannel;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installSchema('cas_user_ban', ['cas_user_ban']);
  }

  /**
   * Test different methods in the service.
   */
  public function testServiceMethods(): void {
    $time_mock = $this->createMock(TimeInterface::class);
    $time_mock->method('getRequestTime')->willReturn(1749809496, 1749809506, 1749809606);
    \Drupal::getContainer()->set('datetime.time', $time_mock);

    $user_manager = \Drupal::service(CasUserBanManagerInterface::class);

    // Check that no entries exist in the banned list.
    $this->assertBannedUsers([]);

    // Check that after calling the add method an entry has been added.
    $user_manager->add('the_joker');
    $this->assertBannedUsers([
      [
        'cas_username' => 'the_joker',
        'timestamp' => '1749809496',
      ],
    ]);

    $log_messages = \Drupal::state()->get('cas_user_ban_test.log_messages', []);
    $this->assertCount(1, $log_messages[LogLevel::NOTICE]);
    $this->assertEquals(
      'The ban for the CAS username the_joker has been added.',
      $log_messages[LogLevel::NOTICE][0],
    );

    // Add more users to check that they are added to the list.
    $user_manager->add('bane');
    $user_manager->add('the_penguin');
    $this->assertBannedUsers([
      [
        'cas_username' => 'the_penguin',
        'timestamp' => '1749809606',
      ],
      [
        'cas_username' => 'bane',
        'timestamp' => '1749809506',
      ],
      [
        'cas_username' => 'the_joker',
        'timestamp' => '1749809496',
      ],
    ]);

    $log_messages = \Drupal::state()->get('cas_user_ban_test.log_messages', []);
    $this->assertCount(3, $log_messages[LogLevel::NOTICE]);
    $this->assertEquals(
      'The ban for the CAS username bane has been added.',
      $log_messages[LogLevel::NOTICE][1],
    );
    $this->assertEquals(
      'The ban for the CAS username the_penguin has been added.',
      $log_messages[LogLevel::NOTICE][2],
    );

    // The user has not been banned hence, the method should return FALSE.
    $this->assertFalse($user_manager->isBanned('batman'));

    // In the other hand users in the list will return TRUE.
    $this->assertTrue($user_manager->isBanned('the_joker'));
    $this->assertTrue($user_manager->isBanned('bane'));
    $this->assertTrue($user_manager->isBanned('the_penguin'));

    // Calling the remove method will shorten the list.
    $user_manager->remove('the_joker');
    $this->assertBannedUsers([
      [
        'cas_username' => 'the_penguin',
        'timestamp' => '1749809606',
      ],
      [
        'cas_username' => 'bane',
        'timestamp' => '1749809506',
      ],
    ]);

    $log_messages = \Drupal::state()->get('cas_user_ban_test.log_messages', []);
    $this->assertCount(4, $log_messages[LogLevel::NOTICE]);
    $this->assertEquals(
      'The ban for the CAS username the_joker has been removed.',
      $log_messages[LogLevel::NOTICE][3],
    );
  }

  /**
   * Empty username on add method will throw an exception.
   */
  public function testEmptyUsername(): void {
    $this->expectExceptionMessage('Username is empty.');
    $this->expectException(InvalidCasUsernameException::class);

    \Drupal::service(CasUserBanManagerInterface::class)->add('');
  }

  /**
   * Un-existing ban on remove method will throw an exception.
   */
  public function testBanNotFound(): void {
    $this->expectExceptionMessage('The ban for the CAS username "batman" could not be found.');
    $this->expectException(BanNotFoundException::class);

    \Drupal::service(CasUserBanManagerInterface::class)->remove('batman');
  }

  /**
   * Existing ban on add method will throw an exception.
   */
  public function testExistingBan(): void {
    $this->expectExceptionMessage('The ban for the CAS username "the_joker" already exists.');
    $this->expectException(ExistingBanException::class);

    \Drupal::service(CasUserBanManagerInterface::class)->add('the_joker');
    \Drupal::service(CasUserBanManagerInterface::class)->add('the_joker');
  }

}
