<?php

namespace Drupal\castorcito_sync\Form;

use Drupal\Core\Archiver\ArchiveTar;
use Drupal\Core\Config\Entity\ConfigEntityType;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Serialization\Yaml;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines the castorcito configuration import form.
 *
 * @internal
 */
class CastorcitoSyncImportForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new ConfigImportForm.
   *
   * @param Drupal\jfu\Services\EntityTypeManagerInterface $entity_type_manager
   *   The The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'castorcito_sync_import_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['markup'] = [
      '#markup' => $this->t('Upload a configuration file package of the components.'),
    ];

    $form['import_tarball'] = [
      '#type' => 'file',
      '#title' => $this->t('Configuration archive'),
      '#description' => $this->t('Allowed types: @extensions.', ['@extensions' => 'tar.gz tgz tar.bz2']),
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Upload'),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $all_files = $this->getRequest()->files->get('files', []);
    if (!empty($all_files['import_tarball'])) {
      $file_upload = $all_files['import_tarball'];
      if ($file_upload->isValid()) {
        $form_state->setValue('import_tarball', $file_upload->getRealPath());
        return;
      }
    }

    $form_state->setErrorByName('import_tarball', $this->t('The file could not be uploaded.'));
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    if ($path = $form_state->getValue('import_tarball')) {
      try {
        $archiver = new ArchiveTar($path, 'gz');
        $files = [];
        foreach ($archiver->listContent() as $file) {
          $filename = $file['filename'];
          $yaml_content = $archiver->extractInString($filename);

          if ($yaml_content) {
            $data = Yaml::decode($yaml_content);
            $config_prefix = $this->getConfigPrefixFromFilename($filename);
            $this->createEntity($config_prefix, $data);
          }
        }

        $this->messenger()->addStatus($this->t('All configurations have been imported.'));
      }
      catch (\Exception $e) {
        $this->messenger()->addError($this->t('Could not extract the contents of the tar file. The error message is <em>@message</em>', ['@message' => $e->getMessage()]));
      }
    }
  }

  /**
   * Creates or updates a configuration entity based on a YAML file.
   *
   * @param string $config_prefix
   * @param array $data
   * @return void
   */
  private function createEntity($config_prefix, array $data) {
    $entity_type = $this->getEntityTypeId($config_prefix);

    if (!empty($entity_type)) {
      $storage = $this->entityTypeManager->getStorage($entity_type);
      $entity = $storage->load($data['id']);

      if (!empty($entity)) {
        foreach ($data as $key => $value) {
          $entity->set($key, $value);
        }
      }
      else {
        $entity = $storage->create($data);
      }

      $entity->save();
    }
  }

  /**
   * Gets the entity type ID based on the config_prefix.
   *
   * @param string $config_prefix
   * @return void
   */
  private function getEntityTypeId($config_prefix) {
    foreach ($this->entityTypeManager->getDefinitions() as $entity_type_id => $definition) {
      if ($definition instanceof ConfigEntityType) {
        if ($definition->getConfigPrefix() === $config_prefix) {
          return $entity_type_id;
        }
      }
    }

    return '';
  }

  /**
   * Extracts the config_prefix from a YML filename
   *
   * @param string $filename
   * @return void
   */
  private function getConfigPrefixFromFilename($filename) {
    $filename = preg_replace('/\.yml$/', '', $filename);
    return substr($filename, 0, strrpos($filename, '.'));
  }

}
