<?php

namespace Drupal\castorcito\Plugin\CastorcitoComponentField;

use Drupal\Core\Form\FormStateInterface;
use Drupal\castorcito\ConfigurableComponentFieldBase;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\InvokeCommand;

/**
 * List Text configuration.
 *
 * @CastorcitoComponentField(
 *   id = "list_text",
 *   label = @Translation("List text"),
 *   description = @Translation("Configuration for the list text field.")
 * )
 */
class ListText extends ConfigurableComponentFieldBase {

  /**
   * {@inheritdoc}
   */
  public function defaultModel() {
    return [
      'value' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return parent::defaultConfiguration() + [
      'options' => [
        [
          'key' => '',
          'label' => '',
        ],
      ],
      'condition' => FALSE,
      'set_default_value' => 0,
      'default_value' => 0,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    if (!$form_state->get('items_count')) {
      $form_state->set('items_count', max(count($this->configuration['options']), 0) - 1);
    }

    if (!$form_state->get('removed_item')) {
      $form_state->set('removed_item', []);
    }

    $form['#tree'] = TRUE;
    $form['field_table_fieldset'] = [
      '#type' => 'details',
      '#title' => $this->t('Add options for the field'),
      '#prefix' => '<div id="names-fieldset-wrapper">',
      '#suffix' => '</div>',
      '#open' => TRUE,
    ];

    $form['field_table_fieldset']['condition'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use as condition field'),
      '#default_value' => $this->configuration['condition'],
      '#description' => '<p>' . $this->t('Use this option to make the field work as a conditional field.') . '</p>',
    ];

    $form['field_table_fieldset']['table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Allowed values'),
        $this->t('Delete'),
        $this->t('Weight'),
      ],
      '#attributes' => [
        'id' => 'allowed-values-order',
        'data-field-list-table' => TRUE,
        'class' => ['allowed-values-table'],
      ],
      '#tabledrag' => [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'weight',
        ],
      ],
    ];

    $max = $form_state->get('items_count');
    $removed_item = $form_state->get('removed_item');
    for ($i = 0; $i <= $max; $i++) {
      if (!in_array($i, $removed_item)) {
        $form['field_table_fieldset']['table'][$i] = [
          '#attributes' => [
            'class' => ['draggable'],
          ],
          '#weight' => $i,
        ];

        $form['field_table_fieldset']['table'][$i]['item'] = [
          'label' => [
            '#type' => 'textfield',
            '#maxlength' => 255,
            '#title' => $this->t('Name'),
            '#default_value' => $this->configuration['options'][$i]['label'],
            '#weight' => -1,
            '#required' => TRUE,
            '#ajax' => [
              'callback' => [$this, 'triggerUpdateValues'],
              'event' => 'change',
            ],
          ],
          'key' => [
            '#type' => 'textfield',
            '#title' => $this->t('Value'),
            '#weight' => 0,
            '#default_value' => $this->configuration['options'][$i]['key'],
            '#required' => TRUE,
            '#ajax' => [
              'callback' => [$this, 'triggerUpdateValues'],
              'event' => 'change',
            ],
          ],
          'state' => [
            '#type' => 'select',
            '#title' => $this->t('Form state'),
            '#default_value' => isset($this->configuration['conditions']) ? $this->configuration['conditions'][$i]['state'] : '',
            '#options' => [
              'visible' => $this->t('Visible'),
              'invisible' => $this->t('Invisible'),
            ],
            '#states'   => [
              'visible' => [
                ':input[name="settings[field_table_fieldset][condition]"]' => ['checked' => TRUE],
              ],
            ],
          ],
          'fields' => [
            '#type' => 'checkboxes',
            '#title' => $this->t('Fields'),
            '#options' => $this->listFields(),
            '#default_value' => isset($this->configuration['conditions']) ? $this->configuration['conditions'][$i]['fields'] : [],
            '#description' => $this->t('If this option is empty, you must add at least one field for conditioning.'),
            '#multiple' => TRUE,
            '#states'   => [
              'visible' => [
                ':input[name="settings[field_table_fieldset][condition]"]' => ['checked' => TRUE],
              ],
            ],
          ],
        ];

        $form['field_table_fieldset']['table'][$i]['delete'] = [
          '#type' => 'submit',
          '#value' => $this->t('Remove'),
          '#name' => $i,
          '#submit' => [
            [$this, 'removeCallback'],
          ],
          '#limit_validation_errors' => [],
          '#ajax' => [
            'callback' => [$this, 'addmoreCallback'],
            'wrapper' => 'names-fieldset-wrapper',
            'effect' => 'fade',
          ],
        ];
        $form['field_table_fieldset']['table'][$i]['weight'] = [
          '#type' => 'weight',
          '#title' => $this->t('Weight for row @number', ['@number' => $i + 1]),
          '#title_display' => 'invisible',
          '#delta' => 50,
          '#default_value' => $i,
          '#attributes' => ['class' => ['weight']],
        ];
      }
    }

    $form['field_table_fieldset']['actions'] = [
      '#type' => 'actions',
    ];

    $form['field_table_fieldset']['actions']['add_name'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add one more'),
      '#submit' => [
        [$this, 'addMoreSubmit'],
      ],
      '#ajax' => [
        'callback' => [$this, 'addmoreCallback'],
        'wrapper' => 'names-fieldset-wrapper',
      ],
    ];

    $form['field_table_fieldset']['actions']['update_values'] = [
      '#type' => 'submit',
      '#value' => $this->t('Update values'),
      '#submit' => [
        [$this, 'updateValuesSubmit'],
      ],
      '#limit_validation_errors' => [],
      '#ajax' => [
        'callback' => [$this, 'setDefaultValueCallback'],
        'wrapper' => 'names-fieldset-wrapper',
      ],
      '#attributes' => [
        'class' => ['js-hide', 'update-values-submit'],
      ],
      '#weight' => 100,
    ];

    $form['field_table_fieldset']['set_default_value'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Set default value'),
      '#default_value' => $this->configuration['set_default_value'],
      '#description' => $this->t('Provide a pre-filled value for the editing form.'),
      '#weight' => 100,
    ];

    if (!$form_state->get('default_options')) {
      $configuration_options = [];
      foreach ($this->configuration['options'] as $value) {
        $configuration_options[$value['key']] = $value['label'];
      }
      $form_state->set('default_options', $configuration_options);
    }
    $default_options = $form_state->get('default_options');
    $form['field_table_fieldset']['default_value'] = [
      '#type' => 'select',
      '#title' => $this->t('Allowed values'),
      '#default_value' => $this->configuration['default_value'],
      '#limit_validation_errors' => [],
      '#options' => $default_options,
      '#states'   => [
        'visible' => [
          ':input[name="settings[field_table_fieldset][set_default_value]"]' => ['checked' => TRUE],
        ],
      ],
      '#weight' => 100,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $field_table_fieldset = $form_state->getValue('field_table_fieldset');

    if (!empty($field_table_fieldset)) {
      $options = [];
      $conditions = [];
      $item = 0;

      foreach ($field_table_fieldset['table'] as $key => $value) {
        $options[$item]['key'] = $value['item']['key'];
        $options[$item]['label'] = $value['item']['label'];
        $conditions[$item]['state'] = $value['item']['state'];
        $conditions[$item]['fields'] = array_filter($value['item']['fields']);
        $item++;
      }

      $this->configuration['options'] = $options;
      $this->configuration['condition'] = $field_table_fieldset['condition'];
      $this->configuration['set_default_value'] = $form_state->getValue(['field_table_fieldset', 'set_default_value']);
      $this->configuration['default_value'] = $form_state->getValue(['field_table_fieldset', 'default_value']);

      if ($field_table_fieldset['condition']) {
        $this->configuration['conditions'] = $conditions;
      }
      else {
        if (isset($this->configuration['conditions'])) {
          unset($this->configuration['conditions']);
        }
      }
    }
  }

  /**
   * Ajax callback for the "Add another item" button.
   */
  public function addmoreCallback(array &$form, FormStateInterface $form_state) {
    return $form['settings']['field_table_fieldset'];
  }

  /**
   * Ajax callback for the "Add another item" button.
   */
  public function setDefaultValueCallback(array &$form, FormStateInterface $form_state) {
    return $form['settings']['field_table_fieldset'];
  }

  /**
   * Ajax callback for trigger button update default values select.
   */
  public function triggerUpdateValues(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    $response->addCommand(new InvokeCommand('.update-values-submit', 'mousedown'));
    return $response;
  }

  /**
   * Adds a new option.
   *
   * @param array $form
   *   The form array to add elements to.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function addMoreSubmit(array &$form, FormStateInterface $form_state) {
    $items_count = $form_state->get('items_count');
    $add_button = $items_count + 1;
    $form_state->set('items_count', $add_button);
    $form_state->setRebuild();
  }

  /**
   * Update default values.
   *
   * @param array $form
   *   The form array to add elements to.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function updateValuesSubmit(array &$form, FormStateInterface $form_state) {
    $default_options = [];
    foreach ($form_state->getUserInput()['settings']['field_table_fieldset']['table'] as $option) {
      $default_options[$option['item']['key']] = $option['item']['label'];
    }
    $form_state->set('default_options', $default_options);
    $form_state->setRebuild();
  }

  /**
   * Deletes a row/option.
   *
   * @param array $form
   *   The form array to add elements to.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function removeCallback(array &$form, FormStateInterface $form_state) {
    $trigger = $form_state->getTriggeringElement();
    $index_remove = $trigger['#name'];
    $removed_fields = $form_state->get('removed_item');
    $removed_fields[] = $index_remove;
    $form_state->set('removed_item', $removed_fields);

    $default_options = [];
    $table_values = $form_state->getUserInput()['settings']['field_table_fieldset']['table'];
    unset($table_values[$index_remove]);
    foreach ($table_values as $option) {
      $default_options[$option['item']['key']] = $option['item']['label'];
    }
    $form_state->set('default_options', $default_options);

    $form_state->setRebuild();
  }

  /**
   * {@inheritdoc}
   */
  protected function listFields() {
    $options = [];

    foreach ($this->component->getComponentFields() as $key => $field) {
      if (!empty($this->getFieldName()) && $field->getFieldName() != $this->getFieldName()) {
        $options[$field->getFieldName()] = $field->getFieldLabel();
      }

      if (empty($this->getFieldName())) {
        $options[$field->getFieldName()] = $field->getFieldLabel();
      }
    }

    return $options;
  }

}
