<?php

namespace Drupal\castorcito;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\Routing\RouteMatchInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a base class for Castorcito component field.
 */
abstract class CastorcitoComponentFieldBase extends PluginBase implements CastorcitoComponentFieldInterface, ContainerFactoryPluginInterface {

  /**
   * The field label.
   *
   * @var string
   */
  protected $label;

  /**
   * The field machine name.
   *
   * @var string
   */
  protected $name;

  /**
   * The human-readable description.
   *
   * @var string|null
   *   The description, or NULL if no description is available.
   */
  protected $description;

  /**
   * Determines whether a data value is required.
   *
   * @var bool
   *   Whether a data value is required.
   */
  protected $required;

  /**
   * Determines whether the field label is hidden.
   *
   * @var bool
   *   Field label is hidden.
   */
  protected $hideLabel;

  /**
   * The castorcito component entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $componentStorage;

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The component entity.
   *
   * @var \Drupal\castorcito\CastorcitoComponentInterface
   */
  protected $component;

  /**
   * Constructs a base class for castorcito component field.
   *
   * @param array $configuration
   *   The plugin configuration array.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\Core\Entity\EntityStorageInterface $component_storage
   *   The castorcito component entity storage.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   Current route match service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityStorageInterface $component_storage, RouteMatchInterface $route_match) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->setConfiguration($configuration);
    $this->componentStorage = $component_storage;
    $this->routeMatch = $route_match;
    $this->component = $this->routeMatch->getParameter('castorcito_component');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')->getStorage('castorcito_component'),
      $container->get('current_route_match'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function label() {
    return $this->pluginDefinition['label'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFieldLabel() {
    return $this->label;
  }

  /**
   * {@inheritdoc}
   */
  public function setFieldLabel($label) {
    $this->label = $label;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getFieldName() {
    return $this->name;
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->description;
  }

  /**
   * {@inheritdoc}
   */
  public function setDescription($description) {
    $this->description = $description;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isRequired() {
    return $this->required;
  }

  /**
   * {@inheritdoc}
   */
  public function setRequired($required) {
    $this->required = $required;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isHideLabel() {
    return $this->hideLabel;
  }

  /**
   * {@inheritdoc}
   */
  public function setHideLabel($isHideLabel) {
    $this->hideLabel = $isHideLabel;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration() {
    return [
      'id' => $this->getPluginId(),
      'label' => $this->getFieldLabel(),
      'name' => $this->getFieldName(),
      'settings' => $this->configuration,
      'description' => $this->getDescription(),
      'required' => $this->isRequired(),
      'hide_label' => $this->isHideLabel(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration) {
    $configuration += [
      'settings' => [],
      'label' => '',
      'name' => '',
      'description' => '',
      'required' => FALSE,
      'hide_label' => FALSE,
    ];
    $this->configuration = $configuration['settings'] + $this->defaultConfiguration();
    $this->label = $configuration['label'];
    $this->name = $configuration['name'];
    $this->description = $configuration['description'];
    $this->required = $configuration['required'];
    $this->hideLabel = $configuration['hide_label'];
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [];
  }

}
