<?php

namespace Drupal\castorcito\Form;

use Drupal\castorcito\CastorcitoManager;
use Drupal\Component\Serialization\Json;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CloseModalDialogCommand;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides a form class to manage allowed values for a Castorcito field list type.
 */
class CastorcitoFieldListTypeAllowedValueForm extends FormBase {

  /**
   * The current HTTP request object.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected $request;

  /**
   * The Component.
   *
   * @var \Drupal\castorcito\CastorcitoComponentInterface
   */
  protected $component;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Castorcito Manager Service.
   *
   * @var \Drupal\castorcito\CastorcitoManager
   */
  protected $castorcitoManager;

  /**
   * Constructs a new CastorcitoFieldListTypeAllowedValueForm.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\castorcito\CastorcitoManager $castorcitoManager
   *   Castorcito manager service.
   */
  public function __construct(RequestStack $request_stack, CastorcitoManager $castorcitoManager) {
    $this->requestStack = $request_stack;
    $this->castorcitoManager = $castorcitoManager;
    $this->request = $this->requestStack->getCurrentRequest();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('request_stack'),
      $container->get('castorcito.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'castorcito_field_list_type_allowed_value_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $component_id = NULL, $field_name = NULL) {
    $this->component = $this->castorcitoManager->castorcitoComponent($component_id);
    $component_fields = $this->getComponentFieldList();
    unset($component_fields[$field_name]);

    if (!$form_state->get('allowed_values_data')) {
      $allowed_values_data = Json::decode($this->request->request->get('allowed_values_data'));
      $form_state->set('allowed_values_data', $allowed_values_data);
    }

    $allowed_values = $form_state->get('allowed_values_data');
    if (!$form_state->get('items_count')) {
      $form_state->set('items_count', max(count($allowed_values['options']), 0));
    }

    $form['allowed_values_data'] = [
      '#type' => 'hidden',
      '#value' => $this->request->request->get('allowed_values_data') ?? '',
    ];

    $form['allowed_values'] = [
      '#type' => 'fieldset',
      '#prefix' => '<div id="cc-allowed-values-wrapper">',
      '#suffix' => '</div>',
    ];

    $form['allowed_values']['condition'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use as condition field'),
      '#default_value' => $allowed_values['condition'] ?? FALSE,
      '#description' => '<p>' . $this->t('Use this option to make the field work as a conditional field.') . '</p>',
      '#disabled' => empty($component_fields),
    ];

    $form['allowed_values']['table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Allowed values'),
        $this->t('Delete'),
        $this->t('Weight'),
      ],
      '#attributes' => [
        'id' => 'allowed-values-order',
        'data-field-list-table' => TRUE,
        'class' => ['ccflt-allowed-values-table'],
      ],
      '#tabledrag' => [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'weight',
        ],
      ],
    ];

    $max = $form_state->get('items_count');
    for ($delta = 0; $delta <= $max; $delta++) {
      $form['allowed_values']['table'][$delta] = [
        '#attributes' => [
          'class' => ['draggable'],
        ],
        '#weight' => $delta,
      ];

      $form['allowed_values']['table'][$delta]['item'] = [
        '#type' => 'details',
        '#title' => $this->t('Item @delta', ['@delta' => $delta + 1]),
        '#open' => TRUE,
        '#weight' => -40,
      ];

      $form['allowed_values']['table'][$delta]['item']['label'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Name'),
        '#weight' => -30,
        '#default_value' => $allowed_values['options'][$delta]['label'] ?? '',
        '#attributes' => [
          'size' => 40,
        ]
      ];

      $form['allowed_values']['table'][$delta]['item']['key'] = [
        '#type' => 'textfield',
        '#maxlength' => 255,
        '#title' => $this->t('Value'),
        '#default_value' => $allowed_values['options'][$delta]['key'] ?? '',
        '#weight' => -20,
        '#attributes' => [
          'size' => 40,
        ]
      ];

      $form['allowed_values']['table'][$delta]['item']['state'] = [
        '#type' => 'select',
        '#title' => $this->t('Form state'),
        '#default_value' => $allowed_values['conditions'][$delta]['state'] ?? '',
        '#options' => [
          'visible' => $this->t('Visible'),
          'invisible' => $this->t('Invisible'),
        ],
        '#states'   => [
          'visible' => [
            ':input[name="condition"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $form['allowed_values']['table'][$delta]['item']['fields'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('Fields'),
        '#options' => $component_fields,
        '#default_value' => $allowed_values['conditions'][$delta]['fields'] ?? [],
        '#description' => $this->t('If this option is empty, you must add at least one field for conditioning.'),
        '#multiple' => TRUE,
        '#states'   => [
          'visible' => [
            ':input[name="condition"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $form['allowed_values']['table'][$delta]['delete'] = [
        '#type' => 'submit',
        '#value' => $this->t('Remove'),
        '#name' => "remove_row_button__$delta",
        '#id' => "remove_row_button__$delta",
        '#delta' => $delta,
        '#submit' => [[static::class, 'deleteSubmit']],
        '#limit_validation_errors' => [],
        '#disabled' => ($max > 0) ? FALSE : TRUE,
        '#ajax' => [
          'callback' => [static::class, 'deleteAjax'],
          'wrapper' => 'cc-allowed-values-wrapper',
          'effect' => 'fade',
        ],
      ];

      $form['allowed_values']['table'][$delta]['weight'] = [
        '#type' => 'weight',
        '#title' => $this->t('Weight for row @number', ['@number' => $delta + 1]),
        '#title_display' => 'invisible',
        '#delta' => 50,
        '#default_value' => 0,
        '#attributes' => ['class' => ['weight']],
      ];
    }

    $form['allowed_values']['add_more_allowed_values'] = [
      '#type' => 'submit',
      '#name' => 'add_more_allowed_values',
      '#value' => $this->t('Add another item'),
      '#attributes' => [
        'class' => ['field-add-more-submit'],
        'data-field-list-button' => TRUE,
      ],
      '#submit' => [[static::class, 'addMoreSubmit']],
      '#ajax' => [
        'callback' => [static::class, 'addMoreAjax'],
        'wrapper' => 'cc-allowed-values-wrapper',
        'effect' => 'fade',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Adding a new item...'),
        ],
      ],
    ];

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
      '#submit' => [],
      '#button_type' => 'primary',
      '#ajax' => [
        'callback' => '::submitForm',
        'event' => 'click',
      ],
    ];

    return $form;
  }

  public function submitForm(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    $allowed_values = $form_state->getValue('table');

    if (!empty($allowed_values)) {
      $item = 0;
      $options_summary = [];
      $data = [
        'options' => [],
        'condition' => $form_state->getValue('condition'),
        'conditions' => [],
      ];

      foreach ($allowed_values as $key => $value) {
        if (!empty($value['item']['key']) && !empty($value['item']['label'])) {
          $hast_option = $this->findOption($data['options'], $value['item']['key']);
          $index = is_null($hast_option) ? $item : $hast_option;

          NestedArray::setValue($data, ['options', $index], [
            'key' => $value['item']['key'],
            'label' => $value['item']['label'],
          ]);

          NestedArray::setValue($data, ['conditions', $index], [
            'state' => $value['item']['state'],
            'fields' => array_filter($value['item']['fields']),
          ]);

          $options_summary[$index] = $value['item']['label'] . ' <small>(' . $value['item']['key'] . ')</small>';

          if (is_null($hast_option)) {
            $item++;
          }
        }
      }

      $response->addCommand(new InvokeCommand('#cc-allowed-values-data-hidden', 'val', [Json::encode($data)]));
      $response->addCommand(new HtmlCommand('#edit-settings-allowed-values .allowed-values-summary', '<ul><li>' . implode('</li><li>', $options_summary) . '</li></ul>'));
      $response->addCommand(new InvokeCommand('.update-values-submit', 'mousedown'));
    }

    $response->addCommand(new CloseModalDialogCommand());
    return $response;
  }

  /**
   * Adds a new option.
   *
   * @param array $form
   *   The form array to add elements to.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public static function addMoreSubmit(array $form, FormStateInterface $form_state) {
    $form_state->set('items_count', $form_state->get('items_count') + 1);
    $form_state->setRebuild();
  }

  /**
   * Ajax callback for the "Add another item" button.
   */
  public static function addMoreAjax(array $form, FormStateInterface $form_state) {
    return $form['allowed_values'];
  }

  /**
   * Deletes a row/option.
   *
   * @param array $form
   *   The form array to add elements to.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public static function deleteSubmit(array $form, FormStateInterface $form_state) {
    $button = $form_state->getTriggeringElement();
    $element = NestedArray::getValue($form, array_slice($button['#array_parents'], 0, -1));
    $user_input = $form_state->getUserInput();
    NestedArray::unsetValue($user_input, $element['#parents']);

    // Reset the keys in the array.
    $table_parents = $element['#parents'];
    array_pop($table_parents);
    $new_values = array_values(NestedArray::getValue($user_input, $table_parents));
    NestedArray::setValue($user_input, $table_parents, $new_values);

    // Reset allowed values data.
    $allowed_values_storage = $form_state->get('allowed_values_data');
    if (!empty($allowed_values_storage)) {
      NestedArray::unsetValue($allowed_values_storage, ['options', end($element['#parents'])]);
      $allowed_values_storage['options'] = array_values($allowed_values_storage['options']);
      $form_state->set('allowed_values_data', $allowed_values_storage);
    }

    $form_state->setUserInput($user_input);
    $form_state->set('items_count', $form_state->get('items_count') - 1);
    $form_state->setRebuild();
  }

  /**
   * Ajax callback for per row delete button.
   */
  public static function deleteAjax(array $form, FormStateInterface $form_state) {
    return $form['allowed_values'];
  }

  /**
   * Returns an associative array of component field names and their labels.
   *
   * @return array
   *   An associative array where the key is the field name and the value is
   *   the field label.
   */
  protected function getComponentFieldList() {
    $options = [];
    foreach ($this->component->getComponentFields() as $key => $field) {
      $options[$field->getFieldName()] = $field->getFieldLabel();
    }

    return $options;
  }

  /**
   * Finds an option by its key in an array of options.
   *
   * @param array $options
   *   The array of options, each option must contain at least ['key'].
   * @param string $search_key
   *   The key to search for within the options.
   *
   * @return int|false
   *   Returns the index of the option if found, or FALSE if not found.
   */
  protected function findOption($options, $search_key) {
    foreach ($options as $index => $option) {
      $value = NestedArray::getValue($options, [$index, 'key'], $key_exists);
      if ($key_exists && $value === $search_key) {
        return $index;
      }
    }

    return NULL;
  }
}
