<?php

namespace Drupal\castorcito\Form;

use Drupal\castorcito\CastorcitoManager;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CloseModalDialogCommand;
use Drupal\Core\Ajax\RedirectCommand;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Field\FieldFilteredMarkup;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides a form to override a single configuration setting of a Castorcito component.
 */
class CastorcitoComponentOverrideConfigForm extends FormBase {

  /**
   * The EntityFormDisplay instance containing form display settings for the specified entity type and bundle.
   */
  protected $formDisplay;

  /**
   * The machine name of the field associated with the entity type and bundle.
   */
  protected $fieldName;

  /**
   * An array of keys specifying the path to the configuration to override within the component.
   */
  protected $configKeys;

  /**
   * An array of settings for the component form display configuration.
   */
  protected $formDisplaySettings;

  /**
   * The current HTTP request object.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected $request;

  /**
   * The entity display repository.
   *
   * @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface
   */
  protected $entityDisplayRepository;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Castorcito Manager Service.
   *
   * @var \Drupal\castorcito\CastorcitoManager
   */
  protected $castorcitoManager;

  /**
   * Constructs a new CastorcitoComponentOverrideConfigForm.
   *
   * @param \Drupal\Core\Entity\EntityDisplayRepositoryInterface $entity_display_repository
   *   The entity display repository.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\castorcito\CastorcitoManager $castorcitoManager
   *   Castorcito manager service.
   */
  public function __construct(EntityDisplayRepositoryInterface $entity_display_repository, RequestStack $request_stack, CastorcitoManager $castorcitoManager) {
    $this->entityDisplayRepository = $entity_display_repository;
    $this->requestStack = $request_stack;
    $this->castorcitoManager = $castorcitoManager;
    $this->request = $this->requestStack->getCurrentRequest();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_display.repository'),
      $container->get('request_stack'),
      $container->get('castorcito.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'castorcito_override_config_form';
  }

  /**
   * {@inheritdoc}
   *
   * @param array $form
   *   A nested array form elements comprising the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   The form structure.
   */
  public function buildForm(array $form, FormStateInterface $form_state, $entity_type = NULL, $bundle = NULL, $field_name = NULL, $config_key = NULL) {
    $this->formDisplay = $this->entityDisplayRepository->getFormDisplay($entity_type, $bundle);
    $this->fieldName = $field_name;
    $this->configKeys = explode('.', $config_key);
    $widget = $this->formDisplay->getComponent($this->fieldName);
    $this->formDisplaySettings = $widget['settings'];

    $last_child_component_config_keys = $this->findLastChildComponent($this->configKeys);
    $component = $this->castorcitoManager->castorcitoComponent($last_child_component_config_keys[0]);

    if (empty($component)) {
      return $form;
    }

    if (!isset($this->formDisplaySettings['components_settings'])) {
      $this->formDisplaySettings['components_settings'] = [];
    }

    // Determine the configuration context (e.g., 'settings' or 'cfields') from the component path.
    $context = $last_child_component_config_keys[1] ?? '';

    // Get the overridden value (if any) from the form display settings using the full configuration key path.
    $overridden_value = NestedArray::getValue($this->formDisplaySettings['components_settings'], $this->configKeys);

    if ($context === 'settings') {
      $form = $this->buildComponentSettingsForm($form, $component, end($last_child_component_config_keys), $overridden_value);
    }

    if ($context === 'cfields') {
      $form = $this->buildComponentCfieldsForm($form, $component, $last_child_component_config_keys, $overridden_value);
    }

    $save_label = $this->t('Save');
    $ccoc_source = $this->request->query->get('ccoc_source');
    if ($ccoc_source === 'use_default') {
      $save_label = $this->t('Use default');
      $field_label = $form[end($last_child_component_config_keys)]['#title'];
      $form[end($last_child_component_config_keys)] = [
        '#markup' => $this->t('<p>Your custom setting for <strong>' . $field_label . '</strong> will be removed. Your component will use the default value. This action cannot be undone. Do you want to continue?</p>'),
      ];
    }

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $save_label,
      '#submit' => [],
      '#button_type' => 'primary',
      '#ajax' => [
        'callback' => '::submitForm',
        'event' => 'click',
      ],
    ];

    $form['actions']['cancel_config'] = [
      '#type' => 'button',
      '#value' => $this->t('Cancel'),
      '#ajax' => [
        'callback' => '::submitCancel',
        'event' => 'click',
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $ccoc_source = $this->request->query->get('ccoc_source');
    if ($ccoc_source === 'use_default') {
      NestedArray::unsetValue($this->formDisplaySettings['components_settings'], $this->configKeys);
    }
    else {
      NestedArray::setValue($this->formDisplaySettings['components_settings'], $this->configKeys, $form_state->getValue(end($this->configKeys)));
    }

    $this->formDisplay->setComponent($this->fieldName, [
      'type' => 'castorcito_component_widget',
      'settings' => $this->formDisplaySettings,
    ])->save();

    $this->messenger()->addMessage($this->t('The configuration was saved successfully.'));
    $response = new AjaxResponse();
    $response->addCommand(new CloseModalDialogCommand());
    $response->addCommand(new RedirectCommand($this->request->headers->get('referer')));

    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function submitCancel(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    $response->addCommand(new CloseModalDialogCommand());

    return $response;
  }

  /**
   * Builds a form element for component setting.
   *
   * @param array $form
   *   The form array to add the component setting to.
   * @param object $component
   *   The component object containing configuration.
   * @param string $cc_field_key
   *   The key for the specific component setting to build (e.g., 'enable_edit_name').
   * @param int|null $overridden_value
   *   The overridden value for the checkbox.
   *
   * @return array
   *   The field element.
   */
  private function buildComponentSettingsForm(array $form, $component, $cc_field_key, $overridden_value) {
    if (is_null($overridden_value)) {
      $form_settings = $component->get('form_settings');
    }

    $form_settings_elements = [
      'enable_edit_name' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable editing of component name'),
        '#description' => $this->t('Checking this option displays a button that allows you to define a component name for better content management.'),
        '#default_value' => !is_null($overridden_value) ? $overridden_value : $form_settings['enable_edit_name'],
      ],
      'collapse_component' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Collapse component, set <strong>close</strong> by default'),
        '#default_value' => !is_null($overridden_value) ? $overridden_value : $form_settings['collapse_component'],
      ],
    ];

    $form[$cc_field_key] = $form_settings_elements[$cc_field_key];

    return $form;
  }

  /**
   * Builds a form element for a component cfield configuration.
   *
   * @param array $form
   *   The form array to add the component cfield configuration to.
   * @param object $component
   *   The component object.
   * @param array $last_child_component_config_keys
   *   An array of keys pointing to the configuration path
   * @param mixed|null $overridden_value
   *   The overridden value to use for the form element.
   *
   * @return array
   *   The form array including the cfield configuration element.
   */
  private function buildComponentCfieldsForm(array $form, $component, $last_child_component_config_keys, $overridden_value) {
    $component_cfield_configurations = $component->get('field_configuration');
    $cfield_config = $component_cfield_configurations[$last_child_component_config_keys[2]];

    if (!in_array('settings', $last_child_component_config_keys)) {
      $form['description'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Help text'),
        '#default_value' => !is_null($overridden_value) ? $overridden_value : $cfield_config['description'],
        '#rows' => 5,
        '#description' => $this->t('Instructions to present to the user below this field on the editing form.<br />Allowed HTML tags: @tags', ['@tags' => FieldFilteredMarkup::displayAllowedTags()]) . '<br />',
      ];
    }
    else {
      $cfield_instance = $this->castorcitoManager->getComponentFieldInstance($cfield_config['id']);
      $cfield_instance_config = $this->castorcitoManager->getComponentFieldInstanceConfigurationForm($cfield_instance);
      $field_key = end($last_child_component_config_keys);
      $form[$field_key] = $cfield_instance_config[$field_key];

      $default_values = !is_null($overridden_value) ? $overridden_value : $cfield_config['settings'][$field_key];
      $this->applyDefaultValuesToElement($form[$field_key], $default_values);

    }

    return $form;
  }

  /**
   * Finds and returns the subpath after the last marker in a configuration path.
   *
   * @param array $parts
   *   The exploded configuration key path.
   *
   * @return array
   *   An array with the keys of the child component.
   */
  private function findLastChildComponent(array $parts) {
    $markers = ['allowed_components', 'allowed_head_component'];
    $last_index = NULL;

    foreach ($parts as $i => $part) {
      if (in_array($part, $markers)) {
        $last_index = $i;
      }
    }

    if (!is_null($last_index)) {
      return array_slice($parts, $last_index + 1);
    }

    return $parts;
  }

  /**
   * Applies default values recursively to a form element.
   *
   * @param array &$element
   *   The form element array to apply default values to.
   * @param array $defaults
   *   The default values to populate the element.
   */
  private function applyDefaultValuesToElement(array &$element, $defaults) {
    if (isset($element['#type']) && $element['#type'] === 'checkboxes') {
      $element['#default_value'] = $defaults;
      return;
    }

    if (isset($element['#default_value']) && !is_array($defaults)) {
      $element['#default_value'] = $defaults;
      return;
    }

    if (is_array($defaults)) {
      foreach ($element as $key => &$value) {
        if (is_array($value) && isset($defaults[$key])) {
          $this->applyDefaultValuesToElement($value, $defaults[$key]);
        }
      }
    }
  }

}
