<?php

namespace Drupal\castorcito\Entity;

use Drupal\castorcito\CastorcitoComponentInterface;
use Drupal\castorcito\CastorcitoComponentFieldPluginCollection;
use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Entity\EntityWithPluginCollectionInterface;
use Drupal\castorcito\CastorcitoComponentFieldInterface;
use Drupal\Component\Serialization\Json;

/**
 * Defines the castorcito component entity type.
 *
 * @ConfigEntityType(
 *   id = "castorcito_component",
 *   label = @Translation("Castorcito component"),
 *   label_collection = @Translation("Castorcito components"),
 *   label_singular = @Translation("castorcito component"),
 *   label_plural = @Translation("castorcito components"),
 *   label_count = @PluralTranslation(
 *     singular = "@count castorcito component",
 *     plural = "@count castorcito components",
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\castorcito\CastorcitoComponentListBuilder",
 *     "form" = {
 *       "add" = "Drupal\castorcito\Form\CastorcitoComponentAddForm",
 *       "edit" = "Drupal\castorcito\Form\CastorcitoComponentEditForm",
 *       "delete" = "Drupal\Core\Entity\EntityDeleteForm",
 *       "clone" = "Drupal\castorcito\Form\CastorcitoComponentCloneForm",
 *     }
 *   },
 *   config_prefix = "component",
 *   admin_permission = "administer castorcito component",
 *   translatable = FALSE,
 *   links = {
 *     "collection" = "/admin/castorcito/component",
 *     "add-form" = "/admin/castorcito/component/add",
 *     "edit-form" = "/admin/castorcito/component/{castorcito_component}",
 *     "delete-form" = "/admin/castorcito/component/{castorcito_component}/delete",
 *     "clone-form" = "/admin/castorcito/component/{castorcito_component}/clone"
 *   },
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "uuid" = "uuid"
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "description",
 *     "model",
 *     "field_configuration",
 *     "inside_container",
 *     "category",
 *     "form_settings",
 *     "display_settings",
 *     "sdc"
 *   }
 * )
 */
class Castorcitocomponent extends ConfigEntityBase implements CastorcitoComponentInterface, EntityWithPluginCollectionInterface {

  /**
   * The id of the component.
   *
   * @var string
   */
  protected $id;

  /**
   * The component label.
   *
   * @var string
   */
  protected $label;

  /**
   * The component description.
   *
   * @var string
   */
  protected $description;

  /**
   * The component model.
   *
   * @var string
   */
  protected $model;

  /**
   * The component configuration.
   *
   * @var array
   */
  protected $field_configuration = [];

  /**
   * The component will only be inside the container.
   *
   * @var bool
   */
  protected $inside_container;

  /**
   * The component category.
   *
   * @var bool
   */
  protected $category;

  /**
   * The component display settings.
   *
   * @var array
   */
  protected $form_settings = [];

  /**
   * The component display settings.
   *
   * @var array
   */
  protected $display_settings = [];

  /**
   * The component SDC.
   *
   * @var array
   */
  protected $sdc = [];

  /**
   * Contains the collection of fields used by this component.
   *
   * @var \Drupal\castorcito\CastorcitoComponentFieldPluginCollection
   */
  protected $fieldsCollection;

  /**
   * {@inheritdoc}
   */
  public function getModel() {
    return !empty($this->model) ? Json::decode($this->model) : [];
  }

  /**
   * {@inheritdoc}
   */
  public function setModel($model) {
    $this->set('model', Json::encode($model));

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setBaseModel($label = '') {
    $model = $this->getModel();
    $model['type'] = $this->id();
    $model['label'] = !empty($label) ? $label : $this->label();
    $this->setModel($model);
  }

  /**
   * {@inheritdoc}
   */
  public function setModelField($name, $field_model) {
    $model = $this->getModel();
    $model['fields'][$name] = $field_model;
    $this->setModel($model);
  }

  /**
   * {@inheritdoc}
   */
  public function setComponentDisplaySettingsValues(array $values) {
    $this->display_settings = array_replace_recursive($this->display_settings, $values);
  }

  /**
   * {@inheritdoc}
   */
  public function updateComponentDisplaySettings(array $settings) {
    $this->set('display_settings', array_merge($this->display_settings, $settings));
    $this->save();
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getComponentField($field) {
    return $this->getComponentFields()->get($field);
  }

  /**
   * {@inheritdoc}
   */
  public function getComponentFields() {
    if (!$this->fieldsCollection) {
      $this->fieldsCollection = new CastorcitoComponentFieldPluginCollection($this->getComponentFieldPluginManager(), $this->field_configuration);
    }

    return $this->fieldsCollection;
  }

  /**
   * {@inheritdoc}
   */
  public function getPluginCollections() {
    return ['field_configuration' => $this->getComponentFields()];
  }

  /**
   * {@inheritdoc}
   */
  public function addComponentField(array $configuration) {
    $this->getComponentFields()->addInstanceId($configuration['name'], $configuration);
    return $configuration['name'];
  }

  /**
   * {@inheritdoc}
   */
  public function deleteComponentField(CastorcitoComponentFieldInterface $field) {
    $model = $this->getModel();
    unset($model['fields'][$field->getFieldName()]);
    $this->setModel($model);
    $this->getComponentFields()->removeInstanceId($field->getFieldName());
    $this->save();
    return $this;
  }

  /**
   * Returns the component field plugin manager.
   *
   * @return \Drupal\Component\Plugin\PluginManagerInterface
   *   The component field plugin manager.
   */
  protected function getComponentFieldPluginManager() {
    return \Drupal::service('plugin.manager.castorcito_component_field');
  }

}
