<?php

namespace Drupal\castorcito\Form;

use Drupal\castorcito\Entity\CastorcitoCategory;
use Drupal\castorcito\CastorcitoManager;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base form for castorcito component add and edit forms.
 */
abstract class CastorcitoComponentFormBase extends EntityForm {

  /**
   * The entity being used by this form.
   *
   * @var \Drupal\castorcito\CastorcitoComponentInterface
   */
  protected $entity;

  /**
   * The castorcito component entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $componentStorage;

  /**
   * Castorcito Manager Service.
   *
   * @var \Drupal\castorcito\CastorcitoManager
   */
  protected $castorcitoManager;

  /**
   * Constructs a base class for castorcito component add and edit forms.
   *
   * @param \Drupal\Core\Entity\EntityStorageInterface $component_storage
   *   The castorcito component entity storage.
   * @param \Drupal\castorcito\CastorcitoManager $castorcitoManager
   *   Castorcito manager service.
   */
  public function __construct(EntityStorageInterface $component_storage, CastorcitoManager $castorcitoManager) {
    $this->componentStorage = $component_storage;
    $this->castorcitoManager = $castorcitoManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')->getStorage('castorcito_component'),
      $container->get('castorcito.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form['component_info'] = [
      '#type' => 'details',
      '#title' => $this->t('Component info'),
      '#open' => FALSE,
    ];

    $form['component_info']['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Component name'),
      '#default_value' => $this->entity->label(),
      '#required' => TRUE,
    ];

    $form['component_info']['id'] = [
      '#type' => 'machine_name',
      '#disabled' => !$this->entity->isNew(),
      '#machine_name' => [
        'exists' => [$this->componentStorage, 'load'],
        'source' => [
          'component_info',
          'label',
        ],
      ],
      '#default_value' => $this->entity->id(),
      '#required' => TRUE,
    ];

    $categories = CastorcitoCategory::loadMultiple();
    $category_options = [];
    if (!empty($categories)) {
      foreach ($categories as $category) {
        $category_options[$category->id()] = $category->label();
      }
    }

    $form['component_info']['category'] = [
      '#type' => 'select',
      '#title' => $this->t('Category'),
      '#options' => $category_options,
      '#default_value' => $this->entity->get('category'),
      '#required' => TRUE,
      '#description' => $this->t('Select the category to which this component belongs.'),
      '#empty_option' => $this->t('- Select a category -'),
    ];

    $form['component_info']['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#default_value' => $this->entity->get('description'),
    ];

    $form['component_info']['inside_container'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('This component is only for use inside a container, hidden in Drupal field widget.'),
      '#default_value' => $this->entity->get('inside_container'),
    ];

    if (!$this->entity->isNew() && $this->castorcitoManager->componentUsage($this->entity->id())) {
      $form['component_info']['inside_container']['#disabled'] = TRUE;
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    parent::save($form, $form_state);
    $form_state->setRedirectUrl($this->entity->toUrl('edit-form'));
  }

}
