<?php

declare(strict_types=1);

namespace Drupal\castorcito\Plugin\rest\resource;

use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\rest\ModifiedResourceResponse;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\file\Entity\File;
use Drupal\file\Validation\FileValidatorInterface;
use Drupal\Core\File\FileExists;
use Drupal\image\Entity\ImageStyle;
use Drupal\Component\Utility\Bytes;
use Drupal\Component\Utility\Environment;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\ByteSizeMarkup;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\HttpException;

/**
 * Represents CastorcitoAjaxUploadImage records as resources.
 *
 * @RestResource (
 *   id = "castorcito_ajax_upload_image_resource",
 *   label = @Translation("Castorcito Ajax Upload Image"),
 *   uri_paths = {
 *     "create" = "/api/castorcito-ajax-upload-image"
 *   }
 * )
 */
final class CastorcitoAjaxUploadImageResource extends ResourceBase {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The file validator service.
   *
   * @var \Drupal\file\FileValidatorInterface
   */
  protected $fileValidator;

  /**
   * Constructs a new CastocitoAjaxUploadImageResource object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param array $serializer_formats
   *   The available serialization formats.
   * @param \Psr\Log\LoggerInterface $logger
   *   A logger instance.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user service.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\file\FileValidatorInterface $file_validator
   *   The file validator service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    array $serializer_formats,
    LoggerInterface $logger,
    AccountProxyInterface $current_user,
    FileSystemInterface $file_system,
    FileValidatorInterface $file_validator,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $serializer_formats, $logger);
    $this->currentUser = $current_user;
    $this->fileSystem = $file_system;
    $this->fileValidator = $file_validator;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->getParameter('serializer.formats'),
      $container->get('logger.factory')->get('rest'),
      $container->get('current_user'),
      $container->get('file_system'),
      $container->get('file.validator')
    );
  }

  /**
   * Responds to POST requests and saves the new record.
   */
  public function post(array $data): ModifiedResourceResponse {
    if (empty($data['image'])) {
      throw new HttpException(400, 'No file provided.');
    }

    $image = base64_decode($data['image']);

    // Define el directorio de destino para guardar la imagen.
    $directory = 'public://' . $data['settings_image']['directory'];
    $this->fileSystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY);
    $destination = $directory . '/' . $data['name'];
    $replace = class_exists('Drupal\Core\File\FileExists') ? FileExists::Rename : $this->fileSystem::EXISTS_RENAME;
    $destination_uri = $this->fileSystem->saveData($image, $destination, $replace);

    // Mueve el archivo cargado a la ubicación pública de Drupal.
    $file = File::create([
      'uri' => $destination_uri,
      'uid' => $this->currentUser->id(),
      'filesize' => $data['size'],
    ]);

    try {
      $response = $this->validateImage($file, $data['settings_image']);
      if (!empty($response['errors'])) {
        return new ModifiedResourceResponse($response, 201);
      }

      $file->setPermanent();
      $file->save();

      $image_factory = \Drupal::service('image.factory');
      $img = $image_factory->get($file->getFileUri());

      $image_style = ImageStyle::load('thumbnail');
      $itok = $image_style->getPathToken($file->getFileUri());
      $thumbnail_uri = $image_style->buildUri($file->getFileUri()) . '?itok=' . $itok;
      $size = ByteSizeMarkup::create($file->getSize());
      $response['data'] = [
        'fid' => $file->id(),
        'name' => $file->getFilename(),
        'size' => $size,
        'value' => $file->getFileUri(),
        'thumbnail' => $thumbnail_uri,
      ];
      // Verificar si el archivo es una imagen válida.
      if ($img->isValid()) {
        $response['data']['width'] = $img->getWidth();
        $response['data']['height'] = $img->getHeight();
      }
    }
    catch (\Exception $e) {
      throw new HttpException(500, 'Error saving file: ' . $e->getMessage());
    }

    return new ModifiedResourceResponse($response, 201);
  }

  /**
   * Callback validate image.
   */
  private function validateImage(File $file, array $settings): array {
    $max_size = $settings['max_size'] ? Bytes::toNumber($settings['max_size']) : Environment::getUploadMaxSize();

    $max_dimensions = 0;
    if (!empty($settings['max_dimensions']['width']) || !empty($settings['max_dimensions']['height'])) {
      $max_dimensions = $settings['max_dimensions']['width'] . 'x' . $settings['max_dimensions']['height'];
    }

    $validators = [
      'FileNameLength' => [],
      'FileSizeLimit' => [
        'fileLimit' => $max_size,
      ],
      'FileExtension' => [
        'extensions' => str_replace(',', '', $settings['extensions']),
      ],
      'FileImageDimensions' => [
        'maxDimensions' => $max_dimensions,
      ],
      // 'FileIsImage' => [],
    ];

    $violations = $this->fileValidator->validate($file, $validators);
    $messages['errors'] = [];
    $messages['status'] = [];
    if ($violations->count() > 0) {
      foreach ($violations as $violation) {
        $messages['errors'][] = $violation->getMessage()->render();
      }
    }
    else {
      $status_messages = $this->messenger()->messagesByType(MessengerInterface::TYPE_STATUS);
      foreach ($status_messages as $message) {
        $messages['status'][] = $message->__toString();
        $this->messenger()->deleteAll();
      }
    }

    return $messages;
  }

}
