<?php

namespace Drupal\castorcito\Form;

use Drupal\castorcito\CastorcitoManager;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CloseModalDialogCommand;
use Drupal\Core\Ajax\RedirectCommand;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Theme\ComponentPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides a form class to override a display configuration of a Castorcito component.
 */
class CastorcitoComponentOverrideDisplayForm extends FormBase {

  /**
   * The EntityViewDisplay instance for managing view display settings of the entity.
   */
  protected $viewDisplay;

  /**
   * The machine name of the field associated with the entity type and bundle.
   */
  protected $fieldName;

  /**
   * An array of keys specifying the path to the configuration to override within the component.
   */
  protected $configKeys;

  /**
   * An array of settings for the component view display configuration.
   */
  protected $viewDisplaySettings;

  /**
   * The current HTTP request object.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected $request;

  /**
   * The entity display repository.
   *
   * @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface
   */
  protected $entityDisplayRepository;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The SDC component plugin manager.
   *
   * @var \Drupal\Core\Theme\ComponentPluginManager;
   */
  protected $componentPluginManager;

  /**
   * Castorcito Manager Service.
   *
   * @var \Drupal\castorcito\CastorcitoManager
   */
  protected $castorcitoManager;

  /**
   * Constructs a new CastorcitoComponentOverrideDisplayForm.
   *
   * @param \Drupal\Core\Entity\EntityDisplayRepositoryInterface $entity_display_repository
   *   The entity display repository.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\Core\Theme\ComponentPluginManager; $component_plugin_manager
   *   The SDC component plugin manager.
   * @param \Drupal\castorcito\CastorcitoManager $castorcitoManager
   *   Castorcito manager service.
   */
  public function __construct(EntityDisplayRepositoryInterface $entity_display_repository, RequestStack $request_stack, ComponentPluginManager $component_plugin_manager, CastorcitoManager $castorcitoManager) {
    $this->entityDisplayRepository = $entity_display_repository;
    $this->requestStack = $request_stack;
    $this->componentPluginManager = $component_plugin_manager;
    $this->castorcitoManager = $castorcitoManager;
    $this->request = $this->requestStack->getCurrentRequest();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_display.repository'),
      $container->get('request_stack'),
      $container->get('plugin.manager.sdc'),
      $container->get('castorcito.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'castorcito_override_display_form';
  }

  /**
   * {@inheritdoc}
   *
   * @param array $form
   *   A nested array form elements comprising the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   The form structure.
   */
  public function buildForm(array $form, FormStateInterface $form_state, $entity_type = NULL, $bundle = NULL, $field_name = NULL, $config_key = NULL) {
    $this->viewDisplay = $this->entityDisplayRepository->getViewDisplay($entity_type, $bundle);
    $this->fieldName = $field_name;
    $this->configKeys = explode('.', $config_key);
    $formatter = $this->viewDisplay->getComponent($this->fieldName);
    $this->viewDisplaySettings = $formatter['settings'];
    $last_child_component_display_keys = $this->findLastChildComponent($this->configKeys);
    $component = $this->castorcitoManager->castorcitoComponent($last_child_component_display_keys[0]);

    if (empty($component)) {
      return $form;
    }

    if (!isset($this->viewDisplaySettings['components_display_settings'])) {
      $this->viewDisplaySettings['components_display_settings'] = [];
    }

    // Determine the configuration context (e.g., 'settings', 'sdc' or 'cfields') from the component path.
    $context = $last_child_component_display_keys[1] ?? '';

    // Get the overridden value (if any) from the view display settings using the full configuration key path.
    $overridden_value = NestedArray::getValue($this->viewDisplaySettings['components_display_settings'], $this->configKeys);

    if ($context === 'display_options') {
      $form = $this->buildComponentDisplaySetting($form, $component, end($last_child_component_display_keys), $overridden_value);
    }

    if ($context === 'sdc') {
      $form = $this->buildComponentSDC($form, $component, end($last_child_component_display_keys), $overridden_value);
    }

    if ($context === 'cfields') {
      $form = $this->buildComponentCfieldsForm($form, $component, $last_child_component_display_keys, $overridden_value);
    }

    $save_label = $this->t('Save');
    $ccod_source = $this->request->query->get('ccod_source');
    if ($ccod_source === 'use_default') {
      $save_label = $this->t('Use default');
      $field_label = $form[end($last_child_component_display_keys)]['#title'];
      $form[end($last_child_component_display_keys)] = [
        '#markup' => $this->t('<p>Your custom setting for <strong>' . $field_label . '</strong> will be removed. Your component will use the default value. This action cannot be undone. Do you want to continue?</p>'),
      ];
    }

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $save_label,
      '#submit' => [],
      '#button_type' => 'primary',
      '#ajax' => [
        'callback' => '::submitForm',
        'event' => 'click',
      ],
    ];

    $form['actions']['cancel_config'] = [
      '#type' => 'button',
      '#value' => $this->t('Cancel'),
      '#ajax' => [
        'callback' => '::submitCancel',
        'event' => 'click',
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $ccod_source = $this->request->query->get('ccod_source');
    if ($ccod_source !== 'use_default') {
      $last_child_component_display_keys = $this->findLastChildComponent($this->configKeys);

      if (in_array('sdc', $last_child_component_display_keys, TRUE)) {
        $sdc_id = $form_state->getValue(end($last_child_component_display_keys));

        if (!$this->componentPluginManager->hasDefinition($sdc_id)) {
          $form_state->setErrorByName(end($last_child_component_display_keys), $this->t('The Single Directory Component ID must follow the format provider:component_id and refer to a valid component.'));
        }
      }

      parent::validateForm($form, $form_state);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();

    if (!$form_state->hasAnyErrors()) {
      $ccod_source = $this->request->query->get('ccod_source');
      if ($ccod_source === 'use_default') {
        NestedArray::unsetValue($this->viewDisplaySettings['components_display_settings'], $this->configKeys);
      }
      else {
        NestedArray::setValue($this->viewDisplaySettings['components_display_settings'], $this->configKeys, $form_state->getValue(end($this->configKeys)));
      }

      $this->viewDisplay->setComponent($this->fieldName, [
        'type' => 'castorcito_component_formatter',
        'settings' => $this->viewDisplaySettings,
      ])->save();

      $this->messenger()->addMessage($this->t('The configuration was saved successfully.'));
    }

    $response->addCommand(new CloseModalDialogCommand());
    $response->addCommand(new RedirectCommand($this->request->headers->get('referer')));

    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function submitCancel(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    $response->addCommand(new CloseModalDialogCommand());

    return $response;
  }

  /**
   * Builds a form element for a specific component display setting.
   *
   * @param array $form
   *   The existing form array to which the component setting will be added.
   * @param object $component
   *   The component object
   * @param string $cc_field_key
   *   The key of the display setting to build.
   * @param int|string|null $overridden_value
   *   The overridden value to apply as the field's default.
   *
   * @return array
   *   The updated form array with the added field.
   */
  private function buildComponentDisplaySetting(array $form, $component, $cc_field_key, $overridden_value) {
    $display_settings = $component->get('display_settings');
    $setting = $display_settings[$cc_field_key];
    $form[$cc_field_key] = [
      '#type' => $setting['type'],
      '#title' => $this->t($setting['label']),
      '#default_value' => !is_null($overridden_value) ? $overridden_value : $setting['default_value'],
      '#description' => $setting['description'] ?? '',
      '#required' => $setting['required'] ?? '',
    ];

    if ($setting['type'] === 'select' && !empty($setting['options'])) {
      $form[$cc_field_key]['#options'] = array_column($setting['options'], 'label', 'key');
    }

    return $form;
  }

  /**
   * Builds a form element for the Single Directory Component (SDC) ID.
   *
   * @param array $form
   *   The existing form array to which the SDC ID field will be added.
   * @param object $component
   *   The component object
   * @param string $cc_field_key
   *   The key of the display setting to build.
   * @param string|null $overridden_value
   *   The overridden SDC ID value to use as the field's default.
   *
   * @return array
   *   The updated form array with the SDC ID field added.
   */
  private function buildComponentSDC(array $form, $component, $cc_field_key, $overridden_value) {
    $default_value = $overridden_value ?? $component->get('sdc')['id'] ?? '';

    $form['id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('SDC ID'),
      '#default_value' => $default_value,
      '#description' => $this->t('Enter the unique identifier for this Single Directory Component, in the format provider:component_id (e.g., my_theme:hero_banner).'),
    ];

    return $form;
  }

  /**
   * Builds a form element for a component cfield configuration.
   *
   * @param array $form
   *   The form array to add the component cfield configuration to.
   * @param object $component
   *   The component object containing cfield configurations.
   * @param array $last_child_component_display_keys
   *   An array of keys pointing to the configuration path
   * @param mixed|null $overridden_value
   *   The overridden value to use for the form element.
   *
   * @return array
   *   The form array including the cfield configuration element.
   */
  private function buildComponentCfieldsForm(array $form, $component, $last_child_component_display_keys, $overridden_value) {
    // Retrieve the field configuration for the target cfield.
    $field_key = end($last_child_component_display_keys);
    $component_cfield_configurations = $component->get('field_configuration');
    $cfield_config = $component_cfield_configurations[$last_child_component_display_keys[2]] ?? [];

    // Load the cfield instance and its configuration form.
    $cfield_instance = $this->castorcitoManager->getComponentFieldInstance($cfield_config['id']);
    $cfield_instance_config = $this->castorcitoManager->getComponentFieldInstanceConfigurationForm($cfield_instance);

    // Get the display settings for the cfield form.
    $elements_display = $cfield_instance_config['display'];
    if ($cfield_config['id'] === 'entity_reference') {
      $entity_type = $cfield_config['settings']['entity_type'];
      $form[$field_key] = $elements_display[$entity_type . '_' . $field_key];
    }
    else {
      $form[$field_key] = $elements_display[$field_key];
    }

    // Determine default values: overridden values > configuration defaults.
    $default_values = is_null($overridden_value) ? $cfield_config['settings']['display'][$field_key] : $overridden_value;
    $this->applyDefaultValuesToElement($form[$field_key], $default_values);

    // If the element is a details container, ensure it is open.
    if ($form[$field_key]['#type'] === 'details') {
      $form[$field_key]['#open'] = TRUE;
    }

    return $form;
  }

  /**
   * Finds the last child component in a configuration path.
   *
   * @param array $parts
   *   The exploded configuration key path (e.g., ['basepack', 'sdc', 'id']).
   *
   * @return array
   *   A sliced array of the path starting from one level above the last
   *   component marker found. Returns the full path if no marker is found.
   */
  private function findLastChildComponent(array $parts) {
    $markers = ['display_options', 'sdc', 'cfields'];
    $last_index = '';

    foreach ($parts as $i => $part) {
      if (in_array($part, $markers)) {
        $last_index = $i;
      }
    }

    return array_slice($parts, $last_index - 1);
  }

  /**
   * Recursively applies default values to a form element.
   *
   * @param array &$element
   *   The form element array to populate with default values.
   * @param array $defaults
   *   The default values to apply. Can be a scalar or nested array.
   */
  private function applyDefaultValuesToElement(array &$element, $defaults) {
    if (!is_array($defaults)) {
      $element['#default_value'] = $defaults;
      return;
    }

    if (is_array($defaults)) {
      foreach ($element as $key => &$value) {
        if (is_array($value) && array_key_exists($key, $defaults)) {
          $this->applyDefaultValuesToElement($value, $defaults[$key]);
        }
      }
    }
  }

}
