(function (Drupal, drupalSettings, once) {
  'use strict';

  // ========================
  // Utils
  // ========================

  // Base URL do Drupal (lida com /web, prefixo de idioma, etc.).
  function buildBaseUrl() {
    const base = (drupalSettings && drupalSettings.path && drupalSettings.path.baseUrl) ? drupalSettings.path.baseUrl : '/';
    const prefix = (drupalSettings && drupalSettings.path && drupalSettings.path.pathPrefix) ? drupalSettings.path.pathPrefix : '';
    return base + prefix; // ex.: "/", "/web/", "/pt-br/"
  }

  // Extrai o prefixo do grupo a partir do name do CEP.
  // Ex.: "...[address][postal_code]" -> retorna tudo antes de "[address]["
  function getPrefixFromPostalName(name) {
    if (!name) return null;
    const marker = '[address][';
    const idx = name.lastIndexOf(marker);
    return idx > -1 ? name.slice(0, idx) : null;
  }

  // Seletor por NOME EXATO (mesmo grupo).
  function selectorByExactName(prefix, key) {
    const full = `${prefix}[address][${key}]`;
    const escaped = full.replace(/"/g, '\\"');
    return `input[name="${escaped}"], select[name="${escaped}"]`;
  }

  // Busca campo do mesmo grupo.
  function findField(prefix, key, container) {
    const sel = selectorByExactName(prefix, key);
    return (container && container.querySelector(sel)) || document.querySelector(sel);
  }

  const sanitizeCep = (v) => (v || '').replace(/\D+/g, '').slice(0, 8);

  function movePostalFirstAndFocus(postalInput) {
    if (!postalInput || postalInput.dataset.cepMoved === '1') return;
    const item = postalInput.closest('.form-item, .js-form-item') || postalInput.parentElement;
    const parent = item && item.parentNode;
    if (item && parent) {
      parent.insertBefore(item, parent.firstChild);
      postalInput.dataset.cepMoved = '1';
      if (!postalInput.value && !postalInput.dataset.cepFocused) {
        postalInput.focus();
        postalInput.dataset.cepFocused = '1';
      }
    }
  }

  // Atualiza rótulo/placeholder do address_line2 para "Complemento".
  function ensureComplementoLabel(prefix, container) {
    const line2 = findField(prefix, 'address_line2', container);
    if (!line2) return;

    if (line2.getAttribute('placeholder') !== 'Complemento') {
      line2.setAttribute('placeholder', 'Complemento');
    }

    const id = line2.getAttribute('id');
    if (id) {
      const label = (container || document).querySelector(`label[for="${CSS.escape(id)}"]`);
      if (label) {
        label.textContent = 'Complemento';
        if (label.classList.contains('visually-hidden')) {
          label.classList.remove('visually-hidden');
        }
      }
    }
    if (!line2.getAttribute('aria-label')) {
      line2.setAttribute('aria-label', 'Complemento');
    }
  }

  // ========================
  // Preenchimento
  // ========================

  // NÃO toca em address_line2 (Complemento).
  function applyFill(prefix, container, json) {
    const data = json && json.data ? json.data : {};
    const raw  = json && json.raw  ? json.raw  : {};

    const postal = findField(prefix, 'postal_code', container);
    const uf     = findField(prefix, 'administrative_area', container);   // Estado (UF)
    const city   = findField(prefix, 'locality', container);              // Cidade
    const bairro = findField(prefix, 'dependent_locality', container);    // Bairro (se existir)
    const line1  = findField(prefix, 'address_line1', container);         // Logradouro

    if (postal) {
      const clean = sanitizeCep(postal.value);
      if (clean && postal.value !== clean) {
        postal.value = clean;
      }
    }

    // Preenche/atualiza SEM disparar eventos (evita loops).
    if (uf)    uf.value    = (data.administrative_area || '');
    if (city)  city.value  = (data.locality || '');
    if (line1) line1.value = (data.address_line1 || '');

    // Bairro (campo próprio).
    const bairroVal = (raw.bairro && raw.bairro.trim()) ? raw.bairro.trim() : '';
    if (bairro) bairro.value = bairroVal;

    // Ajusta label/placeholder do Complemento.
    ensureComplementoLabel(prefix, container);
  }

  async function fetchViaCep(cep) {
    const url = buildBaseUrl() + 'cep-autocomplete/viacep/' + cep;
    const res = await fetch(url, { headers: { 'Accept': 'application/json' } });
    if (!res.ok) throw new Error('ViaCEP indisponível');
    const json = await res.json();
    if (!json || !json.ok || !json.data) throw new Error('CEP inválido/não encontrado');
    return json;
  }

  // ========================
  // Attach (behaviors)
  // ========================

  function attachBehavior(context) {
    const postalInputs = once('cep-autocomplete',
      [
        'input[name*="[address][postal_code]"]',
        'input[name$="[postal_code]"]',
        'input[data-drupal-selector$="postal-code"]'
      ].join(','),
      context
    );

    postalInputs.forEach((postal) => {
      const name = postal.getAttribute('name') || '';
      const prefix = getPrefixFromPostalName(name);
      if (!prefix) return;

      const container = postal.closest('form') || document;

      // Move CEP para o topo e foco inicial.
      movePostalFirstAndFocus(postal);

      // Ajusta o label/placeholder de Complemento no carregamento.
      ensureComplementoLabel(prefix, container);

      let timer;

      const onInput = function () {
        clearTimeout(timer);
        timer = setTimeout(async () => {
          const cep = sanitizeCep(postal.value);
          if (cep.length !== 8) return;

          // Evita reaplicar o mesmo CEP seguidamente.
          if (container.dataset.cepLastApplied === cep) return;

          // Travinha anti-reentrância.
          if (container.dataset.cepApplying === '1') return;
          container.dataset.cepApplying = '1';

          try {
            const json = await fetchViaCep(cep);
            applyFill(prefix, container, json);
            container.dataset.cepLastApplied = cep;
          } catch (e) {
            // silencioso
          } finally {
            container.dataset.cepApplying = '0';
          }
        }, 220);
      };

      postal.addEventListener('input', onInput);
      postal.addEventListener('change', onInput);

      // Completa na carga, se já houver CEP válido.
      const ready = sanitizeCep(postal.value);
      if (ready.length === 8) onInput();
    });
  }

  Drupal.behaviors.cepAutocomplete = {
    attach: function (context) {
      attachBehavior(context);
    }
  };

})(Drupal, drupalSettings, once);
