<?php

namespace Drupal\cep_autocomplete\Controller;

use Drupal\cep_autocomplete\Service\ViaCepClient;
use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * Controller do endpoint de consulta ViaCEP.
 */
class ViaCepController extends ControllerBase {

  public function __construct(
    protected ViaCepClient $client
  ) {}

  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('cep_autocomplete.viacep_client')
    );
  }

  /**
   * Endpoint JSON: /cep-autocomplete/viacep/{cep}
   */
  public function lookup(string $cep): JsonResponse {
    try {
      $raw = $this->client->lookup($cep);

      $mapped = [
        'postal_code' => preg_replace('/\D+/', '', $raw['cep'] ?? $cep),
        'administrative_area' => $raw['uf'] ?? '',
        'locality' => $raw['localidade'] ?? '',
        'address_line1' => $raw['logradouro'] ?? '',
        // Mantemos address_line2 como info complementar, mas o JS não preenche automaticamente.
        'address_line2' => trim(($raw['bairro'] ?? '') . '  ' . ($raw['complemento'] ?? '')),
      ];

      $response = new JsonResponse([
        'ok' => TRUE,
        'data' => $mapped,
        'raw' => $raw,
      ]);
      // Cache HTTP (cliente/proxy). Server-side já é cacheado no serviço.
      $response->setMaxAge(86400);
      $response->setSharedMaxAge(86400);
      return $response;
    }
    catch (\Throwable $e) {
      return new JsonResponse(['ok' => FALSE, 'message' => 'Lookup failed'], 400);
    }
  }

}
