<?php

namespace Drupal\cep_autocomplete\Service;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Cache\CacheBackendInterface;
use GuzzleHttp\ClientInterface;
use Psr\Log\LoggerInterface;

/**
 * Cliente ViaCEP com cache server-side.
 */
class ViaCepClient {

  public function __construct(
    private readonly ClientInterface $httpClient,
    private readonly CacheBackendInterface $cache,
    private readonly TimeInterface $time,
    private readonly LoggerInterface $logger
  ) {}

  /**
   * Busca dados do CEP no ViaCEP (com cache).
   *
   * @param string $cep Somente dígitos (8).
   * @return array Associativo com chaves do ViaCEP.
   */
  public function lookup(string $cep): array {
    $cep = preg_replace('/\D+/', '', $cep ?? '');
    if (strlen($cep) !== 8) {
      throw new \InvalidArgumentException('CEP inválido: esperado 8 dígitos.');
    }

    $cid = 'cep_autocomplete:viacep:' . $cep;
    if ($cache = $this->cache->get($cid)) {
      return $cache->data;
    }

    try {
      $res = $this->httpClient->request('GET', "https://viacep.com.br/ws/{$cep}/json/", [
        'http_errors' => FALSE,
        'timeout' => 5,
        'headers' => [
          'Accept' => 'application/json',
          'User-Agent' => 'Drupal-CEP-Autocomplete/1.0',
        ],
      ]);

      $status = $res->getStatusCode();
      $body = (string) $res->getBody();
      $data = Json::decode($body) ?? [];

      if ($status !== 200 || !is_array($data) || !empty($data['erro'])) {
        throw new \RuntimeException('CEP não encontrado ou ViaCEP indisponível.');
      }

      // Cache por 24h.
      $this->cache->set($cid, $data, $this->time->getRequestTime() + 86400);
      return $data;
    }
    catch (\Throwable $e) {
      $this->logger->warning('Falha ao consultar ViaCEP (@cep): @msg', ['@cep' => $cep, '@msg' => $e->getMessage()]);
      throw new \RuntimeException('Falha ao consultar CEP, tente novamente.');
    }
  }
}
