(function (Drupal, drupalSettings, once) {
  /**
   * Drupal ContentGuide object.
   */
  Drupal.ContentGuide = Drupal.ContentGuide || {};

  /**
   * Behaviors.
   */
  Drupal.behaviors.contentGuide = {
    attach(context, settings) {
      settings.content_guide =
        settings.content_guide || drupalSettings.content_guide;
      if (typeof settings.content_guide === 'undefined') {
        // Return early if settings for Content Guide do not exist.
        return;
      }

      const fields = once('content_guide', '[data-cg-enabled]', context);
      fields.forEach(function (field) {
        const identifier = field.dataset.cgIdentifier || false;
        if (identifier === false) {
          // This should not happen ever, but who knows ...
          return;
        }

        if (!(identifier in settings.content_guide.fields)) {
          // No settings defined for this field.
          return;
        }

        const fieldSettings = settings.content_guide.fields[identifier];
        const fieldSelector =
          fieldSettings.field_selector ||
          `.field--name-${fieldSettings.field_name}`;

        // Remove previously attached documentation.
        Drupal.ContentGuide.detachElement(field);

        const displayType = fieldSettings.display_type;
        // Use dynamic function name to allow other types of display than
        // "tooltip" and "description".
        const functionName = `attach${displayType.charAt(0).toUpperCase()}${displayType.slice(1)}`;

        if (typeof Drupal.ContentGuide[functionName] === 'function') {
          Drupal.ContentGuide[functionName](field, fieldSettings, settings);
        }
      });
    },

    detach(context, settings, trigger) {
      if (trigger === 'unload') {
        return;
      }

      // Find fields that are being removed and clean them up
      const fields = context.querySelectorAll('[data-cg-enabled]');
      fields.forEach(function (field) {
        // Clean up event listeners and FloatingUI
        Drupal.ContentGuide.cleanupField(field);

        // Remove the visual elements
        Drupal.ContentGuide.detachElement(field);

        // Remove once tracking for this field
        once.remove('content_guide', field);
      });
    },
  };

  /**
   * Handle AJAX events to ensure Content Guide is reattached.
   * This uses Drupal's AJAX event system in vanilla JavaScript.
   */
  document.addEventListener('DOMContentLoaded', function () {
    // Listen for Drupal AJAX events
    document.addEventListener('drupalAjaxSuccess', function (event) {
      // After AJAX completes, check for any Content Guide fields that might
      // have been added without proper attachment.
      setTimeout(function () {
        const unprocessedFields =
          document.querySelectorAll('[data-cg-enabled]');
        unprocessedFields.forEach(function (field) {
          // Check if this field has been processed
          const processed = once.find('content_guide', field).length > 0;

          if (!processed) {
            // This field hasn't been processed yet, manually trigger attachment
            const identifier = field.dataset.cgIdentifier;
            if (
              identifier &&
              drupalSettings.content_guide &&
              drupalSettings.content_guide.fields &&
              drupalSettings.content_guide.fields[identifier]
            ) {
              // Mark as processed
              once('content_guide', field);

              const fieldSettings =
                drupalSettings.content_guide.fields[identifier];
              const displayType = fieldSettings.display_type;
              const functionName = `attach${displayType.charAt(0).toUpperCase()}${displayType.slice(1)}`;

              if (typeof Drupal.ContentGuide[functionName] === 'function') {
                Drupal.ContentGuide[functionName](
                  field,
                  fieldSettings,
                  drupalSettings,
                );
              }
            }
          }
        });
      }, 100); // Small delay to ensure DOM is fully updated
    });

    // Also listen to the more generic AJAX complete event
    document.addEventListener('drupalAjaxComplete', function (event) {
      setTimeout(function () {
        const unprocessedFields =
          document.querySelectorAll('[data-cg-enabled]');
        unprocessedFields.forEach(function (field) {
          const processed = once.find('content_guide', field).length > 0;

          if (!processed) {
            const identifier = field.dataset.cgIdentifier;
            if (
              identifier &&
              drupalSettings.content_guide &&
              drupalSettings.content_guide.fields &&
              drupalSettings.content_guide.fields[identifier]
            ) {
              once('content_guide', field);

              const fieldSettings =
                drupalSettings.content_guide.fields[identifier];
              const displayType = fieldSettings.display_type;
              const functionName = `attach${displayType.charAt(0).toUpperCase()}${displayType.slice(1)}`;

              if (typeof Drupal.ContentGuide[functionName] === 'function') {
                Drupal.ContentGuide[functionName](
                  field,
                  fieldSettings,
                  drupalSettings,
                );
              }
            }
          }
        });
      }, 100);
    });
  });

  /**
   * Build the URL to load the content from.
   *
   * @param Node field
   *   HTML element.
   * @param array settings
   *   Drupal settings.
   *
   * @returns string
   *   URL to load the content from.
   */
  Drupal.ContentGuide.buildContentUrl = function (field, settings) {
    const content_url = `${settings.path.baseUrl + settings.path.pathPrefix}content-guide`;

    return content_url;
  };

  /**
   * Clean up event listeners and FloatingUI for a field.
   *
   * @param field
   *   Field to clean up.
   */
  Drupal.ContentGuide.cleanupField = function (field) {
    if (field._cgCleanup && Array.isArray(field._cgCleanup)) {
      field._cgCleanup.forEach(function (cleanupData) {
        // Call FloatingUI cleanup function
        if (typeof cleanupData.cleanup === 'function') {
          cleanupData.cleanup();
        }

        // Remove event listeners
        if (cleanupData.attachElement && cleanupData.eventListeners) {
          cleanupData.eventListeners.forEach(function ([event, listener]) {
            cleanupData.attachElement.removeEventListener(event, listener);
          });
        }
      });

      // Clear the cleanup array
      field._cgCleanup = [];
    }
  };

  /**
   * Remove documentation from the given field.
   *
   * @param field
   *   Field the documentation element is attached to.
   */
  Drupal.ContentGuide.detachElement = function (field) {
    const elements = field.querySelectorAll('[data-cg-content]');
    elements.forEach(function (element) {
      element.remove();
    });
  };
})(Drupal, drupalSettings, once);
