(function (Drupal, drupalSettings) {
  /**
   * Attach a tooltip to the given element.
   *
   * @param field
   *   Element to attach the tooltip to.
   * @param fieldSettings
   *   Field settings.
   * @param settings
   *   Drupal settings.
   */
  Drupal.ContentGuide.attachTooltip = function (
    field,
    fieldSettings,
    settings,
  ) {
    if (field.querySelector('.content-guide--tooltip-icon') !== null) {
      // Icon is already attached to the field.
      return;
    }

    const content_url = Drupal.ContentGuide.buildContentUrl(field, settings);

    fetch(content_url, {
      type: 'get',
      headers: {
        'X-CSRF-Token': fieldSettings._token,
        'X-CG-Identifier': field.dataset.cgIdentifier,
        'X-CG-Document-Path': fieldSettings.document_path,
      },
    })
      .then((response) => response.json())
      .then((data) => renderTooltip(data, field, fieldSettings, settings));

    function renderTooltip(response, field, fieldSettings, settings) {
      const attachSelector =
        fieldSettings.attach_selector || '.form-item__label';

      let attachElement = null;
      if (attachSelector === '::self::') {
        // Use field to attach the tooltip.
        attachElement = field;
      } else {
        attachElement = field.querySelector(attachSelector);
      }

      if (attachElement === null) {
        // Use the first child of the field to attach the tooltip.
        attachElement = field.firstElementChild;
      }

      // Create the required tooltip structure.
      const tooltipElement = document.createElement('div');
      tooltipElement.setAttribute('data-tooltip', true);
      tooltipElement.setAttribute('data-cg-content', true); // Add this for cleanup
      tooltipElement.setAttribute('role', 'tooltip');

      tooltipElement.innerHTML = response.content || response.error;
      const arrowElement = document.createElement('div');
      arrowElement.setAttribute('data-tooltip-arrow', true);
      tooltipElement.appendChild(arrowElement);
      attachElement.after(tooltipElement);

      function update() {
        Drupal.ContentGuide.initFloatingUI(
          attachElement,
          tooltipElement,
          arrowElement,
        );
      }

      function showTooltip() {
        tooltipElement.style.display = 'block';
        update();
      }

      function hideTooltip() {
        tooltipElement.style.display = '';
      }

      // Store event listeners for cleanup
      const eventListeners = [
        ['mouseenter', showTooltip],
        ['mouseleave', hideTooltip],
        ['focus', showTooltip],
        ['blur', hideTooltip],
      ];

      eventListeners.forEach(([event, listener]) => {
        attachElement.addEventListener(event, listener);
      });

      // Start auto updates and store cleanup function
      const cleanup = FloatingUIDOM.autoUpdate(
        attachElement,
        tooltipElement,
        update,
      );

      // Store cleanup data on the field for later removal
      if (!field._cgCleanup) {
        field._cgCleanup = [];
      }
      field._cgCleanup.push({
        cleanup,
        attachElement,
        eventListeners,
      });
    }
  };

  /**
   * Initialize floatingUI for the given elements.
   *
   * @param {Element} attachElement
   *   Element the tooltip is attached to.
   * @param {Element} tooltipElement
   *   The tooltip.
   * @param {Element} arrowElement
   *   Tooltip arrow.
   */
  Drupal.ContentGuide.initFloatingUI = function (
    attachElement,
    tooltipElement,
    arrowElement,
  ) {
    FloatingUIDOM.computePosition(attachElement, tooltipElement, {
      placement: 'top-start',
      middleware: [
        FloatingUIDOM.offset(6),
        FloatingUIDOM.autoPlacement({
          align: 'start',
          autoAlignment: false,
          allowedPlacements: ['top-start', 'bottom-start'],
        }),
        FloatingUIDOM.shift({ padding: 5 }),
        FloatingUIDOM.arrow({ element: arrowElement }),
      ],
    }).then(({ x, y, placement, middlewareData }) => {
      Object.assign(tooltipElement.style, {
        left: `${x}px`,
        top: `${y}px`,
      });

      // Accessing the data
      const { x: arrowX, y: arrowY } = middlewareData.arrow;

      const staticSide = {
        top: 'bottom',
        right: 'left',
        bottom: 'top',
        left: 'right',
      }[placement.split('-')[0]];

      Object.assign(arrowElement.style, {
        left: arrowX !== null ? `${arrowX}px` : '',
        top: arrowY !== null ? `${arrowY}px` : '',
        right: '',
        bottom: '',
        [staticSide]: '-4px',
      });
    });
  };
})(Drupal, drupalSettings);
