<?php

namespace Drupal\cg\Controller;

use Drupal\Component\Utility\UrlHelper;
use Drupal\Component\Utility\Xss;
use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\Core\Cache\CacheableJsonResponse;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Url;
use Drupal\cg\ContentGuideEvents;
use Drupal\cg\EventDispatcher\AlterControllerWidgetSettingsEvent;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Routing\Exception\RouteNotFoundException;

/**
 * Controller for getting taxonomy terms.
 */
class ContentGuideController extends ControllerBase {

  use AutowireTrait;

  /**
   * Content guide configuration.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected $config;

  /**
   * The file system.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Constructs a ContentGuideController object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The current request.
   * @param \Drupal\Core\Access\CsrfTokenGenerator $tokenGenerator
   *   The CSRF token generator service.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   The event dispatcher.
   */
  public function __construct(
    #[Autowire('config.factory')]
    ConfigFactoryInterface $config_factory,
    #[Autowire('entity_type.manager')]
    EntityTypeManagerInterface $entity_type_manager,
    #[Autowire('language_manager')]
    LanguageManagerInterface $language_manager,
    #[Autowire('file_system')]
    FileSystemInterface $file_system,
    #[Autowire('module_handler')]
    ModuleHandlerInterface $module_handler,
    #[Autowire('request_stack')]
    protected RequestStack $requestStack,
    #[Autowire('csrf_token')]
    protected CsrfTokenGenerator $tokenGenerator,
    #[Autowire('event_dispatcher')]
    protected EventDispatcherInterface $eventDispatcher,
  ) {
    $this->configFactory = $config_factory;
    $this->config = $config_factory->get('cg.settings');
    $this->entityTypeManager = $entity_type_manager;
    $this->languageManager = $language_manager;
    $this->fileSystem = $file_system;
    $this->moduleHandler = $module_handler;
  }

  /**
   * Load content guide data.
   *
   * @param string $langcode
   *   (Optional) Language code.
   *
   * @return \Drupal\Core\Cache\CacheableJsonResponse
   *   Cacheable Json response.
   */
  public function getData($langcode = NULL) {
    // Use given langcode or default interface language.
    $langcode_current = $langcode ?? $this->languageManager
      ->getCurrentLanguage(LanguageInterface::TYPE_INTERFACE)
      ->getId();

    $context = [
      'langcode' => $langcode_current,
    ];
    $response = new CacheableJsonResponse($context);

    $request = $this->requestStack->getCurrentRequest();
    $request_is_valid = $this->validateRequest($request, $response);

    if ($request_is_valid === FALSE) {
      return $response;
    }

    /** @var \Symfony\Component\HttpFoundation\Request $request */

    /** @var string $identifier */
    $identifier = $request->headers->get('X-CG-Identifier');
    /** @var string $document_path */
    $document_path = $request->headers->get('X-CG-Document-Path');

    $result = [];
    $settings = [
      'document_path' => $document_path,
    ];

    // Allow modules to alter the settings.
    $context['identifier'] = $identifier;
    $this->moduleHandler->alter('cg_controller_widget_settings', $settings, $context);
    // Use event to alter the settings.
    $alter_controller_widget_settings_event = new AlterControllerWidgetSettingsEvent(
      settings: $settings,
      langcode: $langcode_current,
      identifier: $identifier,
    );
    $this->eventDispatcher->dispatch($alter_controller_widget_settings_event, ContentGuideEvents::ALTER_CONTROLLER_WIDGET_SETTINGS);
    $settings = $alter_controller_widget_settings_event->getSettings();

    if ((count($settings) > 0) && isset($settings['document_path'])) {
      $content = $this->loadDocument($settings['document_path'], $langcode_current);

      $result = (object) [
        'content' => $content,
        'language' => $langcode_current,
      ];
    }

    $response->setData($result);
    $metadata = new CacheableMetadata();
    $metadata->setCacheContexts(['languages:language_interface']);
    $response->addCacheableDependency($metadata);

    return $response;
  }

  /**
   * Validates the request.
   *
   * @param \Symfony\Component\HttpFoundation\Request|null $request
   *   Current request or <code>NULL</code> if there is none.
   * @param \Drupal\Core\Cache\CacheableJsonResponse $response
   *   Response object.
   *
   * @return bool
   *   <code>TRUE</code> for valid requests, <code>FALSE</code> otherwise.
   */
  protected function validateRequest(?Request $request, CacheableJsonResponse $response): bool {
    if (is_null($request)) {
      $response->setData([
        'error' => $this->t('Failed reading request data.'),
      ]);
      return FALSE;
    }

    if (!$request->headers->has('X-CSRF-Token')) {
      $response->setData([
        'error' => $this->t('Missing CSRF token.'),
      ]);
      return FALSE;
    }

    if (!$request->headers->has('X-CG-Identifier')) {
      $response->setData([
        'error' => $this->t('Missing field identifier.'),
      ]);
      return FALSE;
    }

    $token = $request->headers->get('X-CSRF-Token');
    $identifier = $request->headers->get('X-CG-Identifier');

    if (is_null($token) || is_null($identifier) || (strlen($token) === 0) || (strlen($identifier) === 0) || !$this->tokenGenerator->validate($token, $identifier)) {
      $response->setData([
        'error' => $this->t('Invalid CSRF token.'),
      ]);
      return FALSE;
    }

    if (!$request->headers->has('X-CG-Document-Path')) {
      $response->setData([
        'error' => $this->t('Missing document path.'),
      ]);
      return FALSE;
    }

    // No errors. Yeah.
    return TRUE;
  }

  /**
   * Load a document.
   *
   * @param string $path
   *   The path of the document relative to the main content guide location.
   * @param string $langcode
   *   Language code of document to get.
   *
   * @return string|false
   *   The content of the document or FALSE on errors.
   */
  protected function loadDocument($path, $langcode) {
    $document_base_path = \DRUPAL_ROOT . '/' . $this->config->get('document_base_path');
    $base_path = $this->fileSystem->realpath($document_base_path);
    if ($base_path === FALSE) {
      return FALSE;
    }

    $document_path = rtrim($base_path, '/') . '/' . ltrim($path, '/');

    // Strip extension and try loading language specific file.
    $last_dot_position = strrpos($document_path, '.');
    if ($last_dot_position === FALSE) {
      // Malformed document path without extension.
      return FALSE;
    }

    $extension = substr($document_path, $last_dot_position);

    $translated_document_path = substr($document_path, 0, $last_dot_position) . '.' . $langcode . $extension;
    if (file_exists($translated_document_path)) {
      // Use translated document.
      $document_path = $translated_document_path;
    }

    $content = file_get_contents($document_path);

    $parser = new \Parsedown();
    $markup = html_entity_decode($parser->text($content));

    if (strlen($markup) === 0) {
      return FALSE;
    }

    // Find links and try to resolve them to local URLs.
    $links = [];
    preg_match_all('/href="(?P<href>[^"]+)"/', $markup, $links);

    foreach ($links['href'] as $href) {
      if (UrlHelper::isExternal($href)) {
        continue;
      }

      try {
        $url = Url::fromRoute($href)->toString(TRUE);
      }
      catch (RouteNotFoundException $exc) {
        try {
          $url = Url::fromUri($href, ['absolute' => TRUE])->toString(TRUE);
        }
        catch (\InvalidArgumentException $exc) {
          // Do not replace this url.
          continue;
        }
      }

      $markup = strtr($markup, [
        'href="' . $href . '"' => 'href="' . $url->getGeneratedUrl() . '"',
      ]);
    }

    return Xss::filterAdmin($markup);
  }

}
