<?php

namespace Drupal\cg\Controller;

/**
 * @file
 * Ajax Autocomplete Callback for markdown files used by Content Guide.
 */

use Drupal\Component\Utility\Html;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\File\FileSystemInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Returns markdown files in autocomplete.
 *
 * @package Drupal\cg\Controller
 */
class ContentGuideFileAutocompleteController implements ContainerInjectionInterface {

  use AutowireTrait;

  /**
   * Constructs a new autocomplete controller.
   *
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache service.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The file system service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The factory for configuration objects.
   */
  public function __construct(
    #[Autowire('cache.data')]
    protected CacheBackendInterface $cache,
    #[Autowire('file_system')]
    protected FileSystemInterface $fileSystem,
    #[Autowire('config.factory')]
    protected ConfigFactoryInterface $configFactory,
  ) {
  }

  /**
   * Autocomplete for searching markdown files.
   *
   * Searches the configured directory and subdirectory for markdown
   * files to attach them to fields in form display and store them in a cache.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request made from the autocomplete widget.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A json response to use in the autocomplete widget.
   */
  public function autocomplete(Request $request) {
    $response_data = new JsonResponse();

    /** @var string|null $query_string */
    $query_string = $request->query->get('q');
    if (is_null($query_string)) {
      return $response_data;
    }

    $request_filename = Html::escape($query_string);

    // Get the directory of the stored files.
    $config = $this->configFactory->get('cg.settings');
    $document_base_path = $config->get('document_base_path');
    $base_path = $this->fileSystem->realpath(DRUPAL_ROOT . '/' . $document_base_path);
    if ($base_path === FALSE) {
      return $response_data;
    }

    // Get markdown files in configured path. Load from cache if possible.
    $files = $this->getFiles($base_path);

    // Scan all files and return files where the autocomplete entry exists.
    $autocomplete_files = [];
    foreach ($files as $file_parts) {
      // Split the string to only return the subdirectory and filename.
      $file_path = str_replace($base_path, '', $file_parts['dirname']) . '/' . $file_parts['basename'];
      $file_path = ltrim(rtrim($file_path, '/'), '/');
      if (stripos($file_path, $request_filename) === FALSE) {
        continue;
      }
      $autocomplete_files[] = [
        'value' => $file_path,
        'label' => $file_path,
      ];
    }

    $encoded = json_encode($autocomplete_files);
    if ($encoded !== FALSE) {
      $response_data->setContent($encoded);
    }
    return $response_data;
  }

  /**
   * Load markup files from configured path.
   *
   * @param string $path
   *   The path to the markdown files.
   *
   * @return array
   *   A list of markdown files.
   */
  protected function getFiles($path) {
    $cached_files = $this->cache->get('cg_files');
    if (!is_null($cached_files) && ($cached_files !== FALSE)) {
      $data = $cached_files->data;
      assert(is_array($data));
      return $data;
    }
    $files = [];

    // Loop recursively trough the directories and build the autocomplete array.
    $directory_iterator = new \RecursiveDirectoryIterator($path);
    foreach (new \RecursiveIteratorIterator($directory_iterator) as $filename => $file) {
      /** @var string $filename */
      $file_parts = pathinfo($filename);

      if (!isset($file_parts['extension'])) {
        continue;
      }

      if (strlen($file_parts['extension']) === 0) {
        continue;
      }

      switch ($file_parts['extension']) {
        // Get markdown files only.
        case 'md':
          $files[] = $file_parts;
          break;

        default:
          break;
      }
    }
    // Cache files for later usage.
    $this->cache->set(
      'cg_files',
      $files,
      CacheBackendInterface::CACHE_PERMANENT,
      ['content_guide']
    );
    return $files;
  }

}
