<?php

namespace Drupal\cg\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Settings form for Content Guide.
 */
class SettingsForm extends ConfigFormBase {

  use AutowireTrait;

  /**
   * Constructs a \Drupal\cg\Form\SettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The file system service.
   */
  public function __construct(
    #[Autowire('config.factory')]
    ConfigFactoryInterface $config_factory,
    #[Autowire('config.typed')]
    protected TypedConfigManagerInterface $typedConfigManager,
    #[Autowire('file_system')]
    protected FileSystemInterface $fileSystem,
  ) {
    parent::__construct($config_factory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'cg_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['cg.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('cg.settings');

    $form['document_base_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Document base path'),
      '#description' => $this->t('Base path of your Content Guide document. This can either be an absolute path (e.g. <code>/var/www/documentation</code>) or a path relative to your Drupal installation e.g. <code>../private/documentation</code>).'),
      '#default_value' => $config->get('document_base_path'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    $field_name = 'document_base_path';
    /** @var string $document_base_path */
    $document_base_path = $form_state->getValue($field_name);
    $path_absolute = $document_base_path;

    if (!is_dir($path_absolute) || !is_readable($path_absolute)) {
      // Try relative path to document root.
      $path_absolute = rtrim(DRUPAL_ROOT, '/') . '/' . ltrim($path_absolute, '/');
    }

    $real_path = $this->fileSystem->realpath($path_absolute);

    if ($real_path === FALSE) {
      $form_state->setErrorByName(
        $field_name,
        $this->t('The specified path <code>@path</code> does not exist.', ['@path' => $path_absolute])
      );
    }
    elseif (!is_dir($real_path)) {
      $form_state->setErrorByName(
        $field_name,
        $this->t('The specified path <code>@path</code> is a file, not a directory.', ['@path' => $real_path])
      );
    }
    elseif (!is_readable($real_path)) {
      $form_state->setErrorByName(
        $field_name,
        $this->t('The directory <code>@path</code> is not readable by the web server.', ['@path' => $real_path])
      );
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('cg.settings')
      ->set('document_base_path', $form_state->getValue('document_base_path', ''))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
