<?php

namespace Drupal\cg\Form;

use Drupal\cg\ContentGuideEvents;
use Drupal\cg\EventDispatcher\AlterDisplayOptionsEvent;
use Drupal\cg\ThirdPartyHelperInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\Core\Utility\LinkGeneratorInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Build third party helper for content_guide.
 */
class ThirdPartyHelper implements ThirdPartyHelperInterface {

  use StringTranslationTrait;

  /**
   * Constructs a new ThirdPartySettingsFormHelper.
   *
   * @param \Drupal\Core\Utility\LinkGeneratorInterface $linkGenerator
   *   The link generator service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   The event dispatcher.
   */
  public function __construct(
    protected LinkGeneratorInterface $linkGenerator,
    protected ModuleHandlerInterface $moduleHandler,
    protected EventDispatcherInterface $eventDispatcher,
  ) {

  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function buildForm(array $settings): array {
    $settings_link = $this->linkGenerator->generate($this->t('Content Guide'), Url::fromRoute('cg.settings'));

    $element = [
      '#type' => 'fieldset',
      '#title' => $this->t('Content Guide'),
    ];
    $element['document_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Document path'),
      '#description' => $this->t('Enter the path to the content guide document for this field relative to the main @documentation path.', ['@documentation' => $settings_link]),
      '#default_value' => $settings['document_path'] ?? '',
      '#element_validate' => ['cg_validate_document_path'],
      '#autocomplete_route_name' => 'cg.autocomplete_md_files',
    ];

    $element['display_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Display as ...'),
      '#options' => $this->getDisplayOptions(),
      '#default_value' => $settings['display_type'] ?? 'description',
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getSummary(array $settings): array {
    if ((count($settings) === 0) || !isset($settings['document_path'])) {
      return [];
    }

    $summary = [];

    $options = $this->getDisplayOptions();
    $summary[] = $this->t('Content guide: %file (@type)', [
      '%file' => $settings['document_path'],
      '@type' => $options[$settings['display_type']],
    ]);

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getDisplayOptions(): array {
    $options = [
      'tooltip' => $this->t('Tooltip'),
      'description' => $this->t('Field description'),
    ];

    // Let other modules alter the list.
    $this->moduleHandler->alter('cg_display_options', $options);
    // Use event to alter the options.
    $alter_display_options_event = new AlterDisplayOptionsEvent(
      displayOptions: $options,
    );
    $this->eventDispatcher->dispatch($alter_display_options_event, ContentGuideEvents::ALTER_DISPLAY_OPTIONS);

    return $alter_display_options_event->getDisplayOptions();
  }

}
