<?php

namespace Drupal\Tests\cg\Unit\Controller;

use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\Tests\cg\Traits\FileSystemTrait;
use Drupal\cg\Controller\ContentGuideController;
use GuzzleHttp\Utils;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\HeaderBag;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Unit tests for the ContentGuideController.
 */
#[CoversClass(ContentGuideController::class)]
#[Group('cg')]
class ContentGuideControllerTest extends UnitTestCase {

  use FileSystemTrait;

  /**
   * The controller under test.
   */
  protected ContentGuideController $controller;

  /**
   * The relative path to the test directory.
   */
  protected string $testDirectoryRelative = 'cg_docs';

  /**
   * Mock of the ConfigFactory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $configFactory;

  /**
   * Mock of the ImmutableConfig (cg.settings).
   *
   * @var \Drupal\Core\Config\ImmutableConfig|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $config;

  /**
   * Mock of the EntityTypeManager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * Mock of the EntityFormDisplay storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $formDisplayStorage;

  /**
   * Mock of the EntityFormDisplay object.
   *
   * @var \Drupal\Core\Entity\Display\EntityFormDisplayInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $formDisplay;

  /**
   * Mock of the LanguageManager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $languageManager;

  /**
   * Mock of the FileSystem service.
   *
   * @var \Drupal\Core\File\FileSystemInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $fileSystem;

  /**
   * Mock of the ModuleHandler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $moduleHandler;

  /**
   * Mock of the RequestStack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $requestStack;

  /**
   * Mock of the Request object.
   *
   * @var \Symfony\Component\HttpFoundation\Request|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $request;

  /**
   * Mock of the HeaderBag object.
   *
   * @var \Symfony\Component\HttpFoundation\HeaderBag|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $headerBag;

  /**
   * Mock of the CsrfTokenGenerator service.
   *
   * @var \Drupal\Core\Access\CsrfTokenGenerator|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $csrfToken;

  /**
   * Mock of the EventDispatcher service.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $eventDispatcher;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->config = $this->createMock(ImmutableConfig::class);
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->languageManager = $this->createMock(LanguageManagerInterface::class);
    $this->moduleHandler = $this->createMock(ModuleHandlerInterface::class);
    $this->requestStack = $this->createMock(RequestStack::class);
    $this->csrfToken = $this->createMock(CsrfTokenGenerator::class);
    $this->request = $this->createMock(Request::class);
    $this->headerBag = $this->createMock(HeaderBag::class);
    $this->eventDispatcher = $this->createMock(EventDispatcherInterface::class);

    // Setup virtual filesystem.
    $this->setupFileSystem();

    $this->request->headers = $this->headerBag;
    $this->requestStack->method('getCurrentRequest')->willReturn($this->request);

    $this->configFactory->method('get')->with('cg.settings')->willReturn($this->config);

    $langcode_interface = $this->createMock(LanguageInterface::class);
    $langcode_interface->method('getId')->willReturn('en');
    $this->languageManager->method('getCurrentLanguage')->willReturn($langcode_interface);

    $this->eventDispatcher->method('dispatch')->willReturnArgument(0);

    $this->controller = new ContentGuideController(
      $this->configFactory,
      $this->entityTypeManager,
      $this->languageManager,
      $this->fileSystem,
      $this->moduleHandler,
      $this->requestStack,
      $this->csrfToken,
      $this->eventDispatcher
    );

    $translator = $this->createMock(TranslationInterface::class);
    $translator->method('translate')->willReturnCallback(
      function (string $string, array $args = [], array $options = []): string {
        return strtr($string, $args);
      }
    );
    $this->controller->setStringTranslation($translator);
  }

  /**
   * Tests documentation loading.
   *
   * @param string $langcode
   *   Language code.
   * @param string $document_path
   *   Path to document..
   * @param array<string, string> $expected
   *   Expected langcode and content..
   */
  #[DataProvider('documentValuesProvider')]
  public function testGetData(string $langcode, string $document_path, array $expected): void {
    $this->headerBag->method('has')->willReturn(TRUE);
    $this->headerBag->method('get')
      ->willReturnMap([
        ['X-CSRF-Token', NULL, 'VALID_TOKEN'],
        ['X-CG-Identifier', NULL, 'VALID_ID'],
        ['X-CG-Document-Path', NULL, $document_path],
      ]);
    $this->csrfToken->method('validate')->with('VALID_TOKEN', 'VALID_ID')->willReturn(TRUE);

    $response = $this->controller->getData($langcode);
    $content = $response->getContent();

    static::assertIsString($content);
    $data = Utils::jsonDecode($content);

    static::assertInstanceOf(\stdClass::class, $data);
    static::assertObjectHasProperty('content', $data);
    static::assertEquals($expected['langcode'], $data->language);
    static::assertStringContainsString($expected['content'], $data->content);
  }

  /**
   * Provides test cases for translated documents.
   *
   * @return iterable
   *   A list of test cases.
   */
  public static function documentValuesProvider(): iterable {
    yield 'default translation' => [
      'langcode' => 'en',
      'document_path' => 'guide.md',
      'expected' => [
        'langcode' => 'en',
        'content' => '<h1>Hello EN</h1>',
      ],
    ];
    yield 'german translation' => [
      'langcode' => 'de',
      'document_path' => 'guide.md',
      'expected' => [
        'langcode' => 'de',
        'content' => '<h1>Hallo DE</h1>',
      ],
    ];
    yield 'xss filtering' => [
      'langcode' => 'en',
      'document_path' => 'xss.md',
      'expected' => [
        'langcode' => 'en',
        'content' => 'alert(1)Safe',
      ],
    ];
  }

}
