<?php

namespace Drupal\Tests\cg\Unit\Controller;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\Tests\cg\Traits\FileSystemTrait;
use Drupal\cg\Controller\ContentGuideFileAutocompleteController;
use GuzzleHttp\Utils;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\CoversFunction;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Tests ContentGuideFileAutocompleteController.
 */
#[CoversClass('\Drupal\cg\Controller\ContentGuideFileAutocompleteController')]
#[CoversFunction('::autocomplete')]
#[Group('cg')]
class ContentGuideFileAutocompleteControllerTest extends UnitTestCase {

  use FileSystemTrait;

  /**
   * The controller under test.
   *
   * @var \Drupal\cg\Controller\ContentGuideFileAutocompleteController
   */
  protected $controller;

  /**
   * Mock of the file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $fileSystem;

  /**
   * Mock of the config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $configFactory;

  /**
   * Mock of the cache service.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $cache;

  /**
   * Mock of the cg.settings config.
   *
   * @var \Drupal\Core\Config\ImmutableConfig|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $config;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Setup Mocks.
    $this->cache = $this->createMock(CacheBackendInterface::class);
    $this->fileSystem = $this->createMock(FileSystemInterface::class);
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->config = $this->createMock(ImmutableConfig::class);

    // Setup virtual filesystem.
    $this->setupFileSystem();

    // Configure ConfigFactory mock.
    $this->configFactory->method('get')
      ->with('cg.settings')
      ->willReturn($this->config);

    // Instantiate the controller with mocks.
    $this->controller = new ContentGuideFileAutocompleteController(
      $this->cache,
      $this->fileSystem,
      $this->configFactory
    );
  }

  /**
   * Simulates a cache miss for the 'cg_files' key.
   */
  protected function mockCacheMiss(): void {
    $this->cache->method('get')
      ->with('cg_files')
      ->willReturn(FALSE);
  }

  /**
   * Tests autocompletion files.
   *
   * @param string $query
   *   Query string to use for autocompletion.
   * @param string[] $expected
   *   List of files return by autocomplete controller.
   */
  #[DataProvider('autocompleteValuesProvider')]
  public function testAutocomplete(string $query, array $expected): void {
    $this->mockCacheMiss();
    $results = $this->getAutocompleteResults($query);
    static::assertAutocompleteResults($results, $expected);
  }

  /**
   * Provides test cases for different autocomplete values.
   *
   * @return iterable
   *   A list of test cases.
   */
  public static function autocompleteValuesProvider(): iterable {
    yield 'empty query' => [
      'query' => '',
      'expected' => [
        'README.md',
        'altered.md',
        'guide.de.md',
        'guide.md',
        'subdir/ANOTHER.md',
        'subdir/deep/nested.md',
        'xss.md',
      ],
    ];
    yield 'partial query' => [
      'query' => 'ubdi',
      'expected' => [
        'subdir/ANOTHER.md',
        'subdir/deep/nested.md',
      ],
    ];
    yield 'exact match' => [
      'query' => 'README.md',
      'expected' => [
        'README.md',
      ],
    ];
    yield 'nested match' => [
      'query' => 'nested',
      'expected' => [
        'subdir/deep/nested.md',
      ],
    ];
    yield 'no match' => [
      'query' => 'nomatch',
      'expected' => [],
    ];
    yield 'case insensitive' => [
      'query' => 'readme',
      'expected' => [
        'README.md',
      ],
    ];
  }

  /**
   * Helper function to execute the autocomplete request.
   *
   * @param string $query
   *   The value for the 'q' query parameter.
   *
   * @return array
   *   The decoded JSON response data.
   */
  protected function getAutocompleteResults(string $query): array {
    $request = new Request(['q' => $query]);
    $response = $this->controller->autocomplete($request);

    static::assertInstanceOf(JsonResponse::class, $response);
    $content = $response->getContent();
    static::assertNotFalse($content, 'Response content is false.');

    $data = Utils::jsonDecode($content, TRUE);
    static::assertIsArray($data, "JSON decoding failed for query: '{$query}'");

    return $data;
  }

  /**
   * Helper function to assert autocomplete results.
   *
   * @param array $actual_results
   *   The 'value' key from the autocomplete JSON response.
   * @param array $expected_paths
   *   An array of expected file path strings.
   */
  protected function assertAutocompleteResults(array $actual_results, array $expected_paths): void {
    // Extract the 'value' from each result item.
    $actual_values = array_map(fn($item) => $item['value'], $actual_results);

    // Sort both arrays to ensure order doesn't matter for the comparison.
    sort($actual_values);
    sort($expected_paths);

    static::assertEquals($expected_paths, $actual_values);
  }

}
