<?php

declare(strict_types=1);

namespace Drupal\Tests\cg\Unit\Event;

use Drupal\cg\ContentGuideEvents;
use Drupal\cg\EventDispatcher\AlterControllerWidgetSettingsEvent;
use Drupal\cg\EventDispatcher\AlterControllerWidgetSettingsEventInterface;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\CoversMethod;
use PHPUnit\Framework\Attributes\Group;
use Symfony\Component\EventDispatcher\EventDispatcher;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Tests altering display options.
 */
#[CoversClass(AlterControllerWidgetSettingsEvent::class)]
#[CoversMethod(AlterControllerWidgetSettingsEvent::class, 'getDisplayOptions')]
#[Group('cg')]
class AlterControllerWidgetSettingsEventTest extends UnitTestCase {

  /**
   * Ensures a subscriber can alter the controller widget settings.
   */
  public function testSubscriberAltersControllerWidgetSettings(): void {
    // Start with an initial settings.
    $settings = [
      'attach_selector' => '.form-item__label',
      'field_selector' => '.field--name-test',
      'field_name' => 'test',
      'display_type' => 'description',
      'document_path' => 'test.md',
    ];

    // Create event with initial options.
    $event = new AlterControllerWidgetSettingsEvent(
      settings: $settings,
      langcode: 'en',
      identifier: 'test',
    );

    // Register a test subscriber that changes the display type.
    $dispatcher = new EventDispatcher();
    $dispatcher->addSubscriber(new class() implements EventSubscriberInterface {

      /**
       * {@inheritdoc}
       */
      public static function getSubscribedEvents(): array {
        return [
          ContentGuideEvents::ALTER_CONTROLLER_WIDGET_SETTINGS => 'onAlterControllerWidgetSettings',
        ];
      }

      /**
       * Change display type to "tooltip".
       *
       * @param \Drupal\cg\EventDispatcher\AlterControllerWidgetSettingsEventInterface $event
       *   The event carrying the display options array.
       */
      public function onAlterControllerWidgetSettings(AlterControllerWidgetSettingsEventInterface $event): void {
        $settings =& $event->getSettings();
        $settings['display_type'] = 'tooltip';
      }

    });

    $dispatcher->dispatch($event, ContentGuideEvents::ALTER_CONTROLLER_WIDGET_SETTINGS);

    // Verify the event contains the changed display type.
    $settings = $event->getSettings();
    self::assertSame('tooltip', $settings['display_type']);
  }

}
