<?php

declare(strict_types=1);

namespace Drupal\Tests\cg\Unit\Event;

use Drupal\cg\ContentGuideEvents;
use Drupal\cg\EventDispatcher\AlterDisplayOptionsEvent;
use Drupal\cg\EventDispatcher\AlterDisplayOptionsEventInterface;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\CoversMethod;
use PHPUnit\Framework\Attributes\Group;
use Symfony\Component\EventDispatcher\EventDispatcher;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Tests altering display options.
 */
#[CoversClass(AlterDisplayOptionsEvent::class)]
#[CoversMethod(AlterDisplayOptionsEvent::class, 'getDisplayOptions')]
#[Group('cg')]
class AlterDisplayOptionsEventTest extends UnitTestCase {

  /**
   * Ensures a subscriber can alter the list of display options.
   */
  public function testSubscriberAltersDisplayOptions(): void {
    // Start with an initial set of options.
    $initial = [
      'alpha' => 'Alpha',
      'beta' => 'Beta',
    ];

    // Create event with initial options.
    $event = new AlterDisplayOptionsEvent($initial);

    // Register a test subscriber that adds one extra option.
    $dispatcher = new EventDispatcher();
    $dispatcher->addSubscriber(new class() implements EventSubscriberInterface {

      /**
       * {@inheritdoc}
       */
      public static function getSubscribedEvents(): array {
        return [
          ContentGuideEvents::ALTER_DISPLAY_OPTIONS => 'onAlterDisplayOptions',
        ];
      }

      /**
       * Adds an extra option to the display options list.
       *
       * @param \Drupal\cg\EventDispatcher\AlterDisplayOptionsEventInterface $event
       *   The event carrying the display options array.
       */
      public function onAlterDisplayOptions(AlterDisplayOptionsEventInterface $event): void {
        $options =& $event->getDisplayOptions();
        $options['gamma'] = 'Gamma';
      }

    });

    $dispatcher->dispatch($event, ContentGuideEvents::ALTER_DISPLAY_OPTIONS);

    // Verify the event contains the added option while retaining originals.
    $result = $event->getDisplayOptions();
    self::assertArrayHasKey('alpha', $result);
    self::assertArrayHasKey('beta', $result);
    self::assertArrayHasKey('gamma', $result);
    self::assertSame('Gamma', $result['gamma']);
  }

}
